<?php

namespace Mtc\MercuryDataModels\Services;

use App\Facades\Settings;
use Mtc\MercuryDataModels\Contracts\LocatingServiceEngine;
use Mtc\MercuryDataModels\PostcodeCache;

class LocatingService
{
    private $location;

    private array $locations = [];

    public function __construct(private readonly LocatingServiceEngine $engine)
    {
    }

    public function locate(string $postCode): self
    {
        $this->location = $this->localCopy($this->normalize($postCode));
        if (!$this->location) {
            $this->location = $this->engine->find($this->normalize($postCode));
            $this->cacheLocally($this->normalize($postCode));
        }
        if (!$this->location) {
            throw new \Exception('Could not locate coordinates for ' . $postCode);
        }
        return $this;
    }

    public function lat(): ?string
    {
        return $this->location?->latitude;
    }

    public function lng(): ?string
    {
        return $this->location?->longitude;
    }

    private function normalize(string $postCode): string
    {
        return strtoupper(str_replace(' ', '', $postCode));
    }

    public function asString(): string
    {
        return  $this->lat() . ',' . $this->lng();
    }

    private function localCopy(string $postCode): ?object
    {
        if (!isset($this->locations[$postCode])) {
            $this->locations[$postCode] = PostcodeCache::query()->where('postcode', $postCode)->first();
        }

        return $this->locations[$postCode];
    }

    private function cacheLocally(string $postcode): void
    {
        if ($this->lng()) {
            $this->locations[$postcode] = PostcodeCache::query()
                ->create([
                    'country' => Settings::get('app-details-country') ?? config('app.default_country'),
                    'postcode' => $postcode,
                    'lat' => $this->location->latitude,
                    'lng' => $this->location->longitude,
                ]);
        }
    }
}
