<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;

class IsVatApplicableFilter extends IsFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selections = [])
    {
        if (!$this->filterIsApplicable()) {
            return;
        }

        $query->where(function ($query) use ($selections) {
            foreach ($selections as $selection) {
                if (is_bool($selection)) {
                    $query->orWhere('is_vat_applicable', $selection ? 1 : 0);
                }
            }
        });
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        if (!$this->filterIsApplicable()) {
            return collect();
        }

        $query = $this->getQuery()->where($product_filtering)->active();

        return collect([
            [
                'id'     => 'is_vat_applicable',
                'name'   => 'Yes',
                'count'  => $query->clone()->where('is_vat_applicable', 1)->count(),
                'slug'   => 'true',
            ],
            [
                'id'     => 'is_not_vat_applicable',
                'name'   => 'No',
                'count'  => $query->clone()->where('is_vat_applicable', 0)->count(),
                'slug'   => 'false',
            ],
        ])
        ->filter(fn($item) => !empty($item['count']))
        ->values();
    }


    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => Settings::get('automotive-vehicle-filters-is_vat_applicable') ?? true,
            'has_load_more' => $this->hasLoadMore($collection),
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'results' => $collection
        ];
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return $this->filter_target_model;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'VAT Applicable';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->is_vat_applicable ? 'true' : 'false';
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return 'is_vat_applicable';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'is_vat_applicable';
    }

    /**
     * Whether this filter is visible by default
     * @return bool
     */
    public function visibleByDefault(): bool
    {
        return false;
    }

    private function getFormattedName(string $name): string
    {
        return 'Is VAT Applicable';
    }

    public function filterType(): string
    {
        return 'is_vat_applicable';
    }
}
