<?php

namespace Mtc\MercuryDataModels\Finance\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;
use Mtc\MercuryDataModels\Finance\FinanceServiceHelper;

class FetchCapBasedVehicleFinanceData implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(
        protected readonly string $cap_id,
        protected readonly int $price,
        protected readonly int $term,
        protected readonly int $annual_mileage,
        protected readonly int $deposit,
        protected readonly string $credit_rating,
        protected readonly array $extra,
        protected readonly bool $store_result = true,
    ) {
        $this->onQueue('finance');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle(): FinanceResultCollection
    {
        return $this->fetch($this->prepare());
    }

    /**
     * Prepare data for fetching
     *
     * @return FinanceRequestData
     */
    protected function prepare(): FinanceRequestData
    {
        return new FinanceRequestData(
            uuid: $this->cap_id,
            registration_number: '',
            cap_id: $this->cap_id,
            dealer_id: '',
            engine_size: null,
            condition: 'new',
            registration_date: null,
            term: $this->term,
            mileage: 0,
            annual_mileage: $this->annual_mileage,
            price: $this->price,
            deposit: $this->deposit,
            credit_rating: $this->credit_rating,
            clientKey: null,
            finance_option: null,
            extra: $this->extra,
            is_new: true,
            make_name: null,
        );
    }

    /**
     * Fetch data from service
     *
     * @param FinanceRequestData $request
     * @return FinanceResultCollection
     */
    protected function fetch(FinanceRequestData $request): FinanceResultCollection
    {
        return $this->getFinanceProvider()->calculate($request);
    }

    /**
     * Gert the finance provider for client
     *
     * @return FinanceProvider
     */
    protected function getFinanceProvider(): FinanceProvider
    {
        return FinanceServiceHelper::initializeForSite();
    }
}
