<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Mtc\Crm\Traits\ModelSortAndFilter;
use Mtc\Filter\FilterIndex;
use Mtc\MercuryDataModels\Factories\BodyStyleTypeFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\OrderByName;

class BodyStyleType extends Model
{
    use HasFactory;
    use ModelSortAndFilter;
    use EnsuresSlug;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'body_styles';

    /**
     * Mass assign columns
     *
     * @var string[]
     */
    protected $fillable = [
        'id',
        'name',
        'slug',
        'autotrader_id',
        'has_subcategories',
        'small_if_under',
        'large_if_above',
    ];

    /**
     * Columns shown in JSON response by default
     *
     * @var string[]
     */
    protected $visible = [
        'id',
        'name',
        'slug',
    ];

    protected static function boot()
    {
        parent::boot();

        self::addGlobalScope(fn($query) => $query->orderBy('order')->orderBy('name'));

        self::saving(function (self $bodyStyleType) {
            $bodyStyleType->ensureSlug();
        });
    }

    /**
     * Model factory
     *
     * @return BodyStyleTypeFactory
     */
    protected static function newFactory()
    {
        return BodyStyleTypeFactory::new();
    }

    /**
     * Relationship with vehicles
     *
     * @return HasMany
     */
    public function vehicles(): HasMany
    {
        return $this->hasMany(Vehicle::class, 'body_style_id');
    }

    /**
     * Relationship with offers
     *
     * @return HasMany
     */
    public function offers(): HasMany
    {
        return $this->hasMany(VehicleOffer::class, 'body_style_id');
    }

    /**
     * Relationship with new cars
     *
     * @return HasMany
     */
    public function newCars(): HasMany
    {
        return $this->hasMany(NewCar::class, 'body_style_id');
    }

    public function filterIndex(): MorphOne
    {
        return $this->morphOne(FilterIndex::class, 'filter');
    }

    protected function getSlugAttribute(): string
    {
        if (empty($this->attributes['slug'])) {
            return $this->filterIndex?->slug ?? '';
        }
        return $this->attributes['slug'];
    }
}
