<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Vehicle;

class AgeFilter extends IndexedFilter implements CustomPatternFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return mixed
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereIn('manufacture_year', $selection);
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return Vehicle::query()
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->distinct()
            ->orderByDesc('manufacture_year')
            ->pluck('manufacture_year')
            ->map(fn ($year) => [
                'id' => $year,
                'name' => $year,
            ]);
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return '';
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Age';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->name;
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
        ];
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/year-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/year-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'year-' . $selection;
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Year ' . $selection;
    }

    public function filterType(): string
    {
        return 'age';
    }
}
