<?php

namespace Mtc\MercuryDataModels;

use App\Master\HasProfilePhoto;
use App\Models\ReleaseNote;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\Factories\UserFactory;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Contracts\Role;
use Spatie\Permission\PermissionRegistrar;
use Spatie\Permission\Traits\HasRoles;
use Stancl\Tenancy\Database\Concerns\CentralConnection;

class User extends Authenticatable
{
    use CentralConnection;
    use HasApiTokens;
    use HasFactory;
    use Notifiable;
    use TwoFactorAuthenticatable;
    use HasProfilePhoto;
    use HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var string[]
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'requires_password_change'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'updated_at',
        'created_at',
    ];

    /**
     * Append attributes to response
     *
     * @var string[]
     */
    protected $appends = [
        'two_factor_enabled',
        'profile_photo_url'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'release_notes_checked_at' => 'datetime',
        'requires_password_change' => 'boolean',
    ];

    /**
     * Create a new factory instance for the model.
     *
     * @return Factory
     */
    protected static function newFactory()
    {
        return UserFactory::new();
    }

    public function tenants(): BelongsToMany
    {
        return $this->belongsToMany(Tenant::class, 'tenant_users')
            ->withPivot([
                'role',
            ]);
    }

    public function dashboardWidgets(): HasMany
    {
        return $this->hasMany(UserDashboardWidget::class, 'user_id');
    }

    public function getTwoFactorEnabledAttribute()
    {
        return !empty($this->attributes['two_factor_secret']);
    }

    /**
     * Determine if the model has (one of) the given role(s).
     *
     * @param string|int|array|\Spatie\Permission\Contracts\Role|\Illuminate\Support\Collection $roles
     * @param string|null $guard
     * @return bool
     */
    public function hasRole($roles, string $guard = null): bool
    {
        if (tenant('id')) {
            setPermissionsTeamId(tenant('id'));
        }

        $this->loadMissing('roles');

        if (is_string($roles) && false !== strpos($roles, '|')) {
            $roles = $this->convertPipeToArray($roles);
        }

        if (is_string($roles)) {
            return $guard
                ? $this->roles->where('guard_name', $guard)->contains('name', $roles)
                : $this->roles->contains('name', $roles);
        }

        if (is_int($roles)) {
            $roleClass = $this->getRoleClass();
            $key = (new $roleClass())->getKeyName();

            return $guard
                ? $this->roles->where('guard_name', $guard)->contains($key, $roles)
                : $this->roles->contains($key, $roles);
        }

        if ($roles instanceof Role) {
            return $this->roles->contains($roles->getKeyName(), $roles->getKey());
        }

        if (is_array($roles)) {
            foreach ($roles as $role) {
                if ($this->hasRole($role, $guard)) {
                    return true;
                }
            }

            return false;
        }

        return $roles->intersect($guard ? $this->roles->where('guard_name', $guard) : $this->roles)->isNotEmpty();
    }

    /**
     * A model may have multiple roles.
     */
    public function roles(): BelongsToMany
    {
        $relation = $this->morphToMany(
            config('permission.models.role'),
            'model',
            config('permission.table_names.model_has_roles'),
            config('permission.column_names.model_morph_key'),
            PermissionRegistrar::$pivotRole
        );

        if (! PermissionRegistrar::$teams) {
            return $relation;
        }

        return $relation->where(function ($q) {
            $teamField = config('permission.table_names.model_has_roles') . '.' . PermissionRegistrar::$teamsKey;
            $q->where($teamField, '')->orWhere($teamField, getPermissionsTeamId());
        });
    }

    /**
     * Return all the permissions the model has via roles.
     */
    public function getPermissionsViaRoles(): Collection
    {
        return $this->roles()
            ->with('permissions')
            ->get()
            ->flatMap(function ($role) {
                return $role->permissions;
            })->sort()->values();
    }

    public function hasUnreadReleaseNotes(): bool
    {
        return ReleaseNote::query()
            ->when(
                !empty($this->release_notes_checked_at),
                fn($query) => $query->where('published_at', '>', $this->release_notes_checked_at)
            )
            ->published()
            ->exists();
    }

    public function hasSeenReleaseNote(ReleaseNote $note): bool
    {
        if (empty($this->release_notes_checked_at)) {
            return false;
        }
        return $note->published_at?->lte($this->release_notes_checked_at) ?? true;
    }

    public function markSeenReleaseNote(ReleaseNote $note): void
    {
        if ($note->published_at && $note->published_at->lt($this->release_notes_checked_at)) {
            // more recent post already marked as seen
            return;
        }

        $this->release_notes_checked_at = $note->published_at;
        $this->save();
    }

    /**
     * Get the default profile photo URL if no profile photo has been uploaded.
     *
     * @return string
     */
    protected function defaultProfilePhotoUrl()
    {
        if (!empty($this->attributes['name'])) {
            return 'https://ui-avatars.com/api/?name=' . urlencode($this->name) . '&color=7F9CF5&background=EBF4FF';
        }

        return 'https://ui-avatars.com/api/?name=' . urlencode($this->email) . '&color=7F9CF5&background=EBF4FF';
    }
}
