<?php

namespace Mtc\MercuryDataModels;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\Factories\KeyloopLeaseVehicleVariantFactory;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\HasVehiclePackages;

class KeyloopLeaseVehicleVariant extends Model
{
    use HasFactory;
    use EnsuresSlug;
    use HasVehiclePackages;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'keyloop_lease_vehicle_variants';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'external_variant_id',
        'slug',
        'make_id',
        'model_id',
        'name',
        'is_current_variant',
        'is_cheapest_variant',
        'type',
        'body_style_id',
        'transmission_id',
        'fuel_type_id',
        'engine_size_cc',
        'brake_horsepower',
        'co2',
        'door_count',
        'seats',
        'road_fund_licence',
        'was_recently_synced',
        'price_ex_vat_ex_vrt',
        'price_inc_vat_ex_vrt',
        'vehicle_vrt',
        'price_inc_vat_inc_vrt',
        'delivery_ex_vat_ex_vrt',
        'delivery_inc_vat_ex_vrt',
        'delivery_vrt',
        'delivery_inc_vat_inc_vrt',
        'featured',
        'finance_recently_updated',
        'cheapest_monthly_price_ex_vat',
        'cheapest_monthly_price_inc_vat',
    ];

    protected $casts = [
        'is_cheapest_variant' => 'boolean',
        'is_current_variant' => 'boolean',
        'was_recently_synced' => 'boolean',
        'featured' => 'boolean',
        'finance_recently_updated' => 'boolean',
    ];

    protected $appends = [
        'derivative',
    ];

    protected static function newFactory()
    {
        return KeyloopLeaseVehicleVariantFactory::new();
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function make(): BelongsTo
    {
        return $this->belongsTo(VehicleMake::class, 'make_id');
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function model(): BelongsTo
    {
        return $this->belongsTo(VehicleModel::class, 'model_id');
    }

    /**
     * Relationship with fuel type
     *
     * @return BelongsTo
     */
    public function fuelType(): BelongsTo
    {
        return $this->belongsTo(FuelType::class, 'fuel_type_id');
    }

    /**
     * Relationship with transmission
     *
     * @return BelongsTo
     */
    public function transmission(): BelongsTo
    {
        return $this->belongsTo(TransmissionType::class, 'transmission_id');
    }

    /**
     * Relationship with bodyStyle
     *
     * @return BelongsTo
     */
    public function bodyStyle(): BelongsTo
    {
        return $this->belongsTo(BodyStyleType::class, 'body_style_id');
    }

    public function options(): HasManyThrough
    {
        return $this->hasManyThrough(
            KeyloopLeaseOption::class,
            KeyloopLeaseVariantOptionMapping::class,
            'external_variant_id',
            'option_code',
            'external_variant_id',
            'option_code',
        )
            ->whereRaw('keyloop_lease_variant_option_mapping.model_id = keyloop_lease_options.model_id')
            ->leftJoin(
                'keyloop_lease_option_price_mapping',
                'keyloop_lease_option_price_mapping.price_option_code',
                '=',
                'keyloop_lease_options.option_code',
            )
            ->where(
                'keyloop_lease_variant_option_mapping.option_availability_code',
                '=',
                KeyloopFleetKompactOptionAvailabilityType::OPTIONAL
            )
            ->whereIn('keyloop_lease_variant_option_mapping.option_type', [
                KeyloopFleetKompactOptionType::FACTORY_OPTION,
                KeyloopFleetKompactOptionType::PACK,
            ])
            ->whereIn('keyloop_lease_options.option_type', [
                KeyloopFleetKompactOptionType::FACTORY_OPTION,
                KeyloopFleetKompactOptionType::PACK,
            ])
            // valid options should not have price mappings
            ->whereNull('keyloop_lease_option_price_mapping.id');
    }

    public function spec(): HasManyThrough
    {
        return $this->hasManyThrough(
            KeyloopLeaseOption::class,
            KeyloopLeaseVariantOptionMapping::class,
            'external_variant_id',
            'option_code',
            'external_variant_id',
            'option_code',
        )
            ->whereRaw('keyloop_lease_variant_option_mapping.model_id = keyloop_lease_options.model_id')
            ->where('option_availability_code', '=', KeyloopFleetKompactOptionAvailabilityType::STANDARD)
            ->where(
                'keyloop_lease_variant_option_mapping.option_type',
                '=',
                KeyloopFleetKompactOptionType::FACTORY_OPTION
            );
    }

    public function getTrims(): Collection
    {
        return $this->trimsRelation()
            ->get()
            ->map(function ($trim) {
                $trim['price'] = $trim->getPrice($this->external_variant_id);
                return $trim;
            });
    }

    public function trimsRelation(): HasManyThrough
    {
        return $this->hasManyThrough(
            KeyloopLeaseOption::class,
            KeyloopLeaseVariantOptionMapping::class,
            'external_variant_id',
            'option_code',
            'external_variant_id',
            'option_code',
        )
            ->whereRaw('keyloop_lease_variant_option_mapping.model_id = keyloop_lease_options.model_id')
            ->where('option_availability_code', '=', KeyloopFleetKompactOptionAvailabilityType::OPTIONAL)
            ->where('keyloop_lease_variant_option_mapping.option_type', '=', KeyloopFleetKompactOptionType::TRIM)
            ->where('keyloop_lease_options.option_type', '=', KeyloopFleetKompactOptionType::TRIM);
    }

    public function getColours(): Collection
    {
        return $this->coloursRelation()
            ->get()
            ->map(function ($colour) {
                $colour['price'] = $colour->getPrice($this->external_variant_id);
                return $colour;
            });
    }

    private function coloursRelation(): HasManyThrough
    {
        return $this->hasManyThrough(
            KeyloopLeaseColour::class,
            KeyloopLeaseVariantOptionMapping::class,
            'external_variant_id',
            'colour_code',
            'external_variant_id',
            'option_code',
        )
            ->whereRaw('keyloop_lease_variant_option_mapping.make_id = keyloop_lease_colours.make_id')
            ->where(
                'keyloop_lease_variant_option_mapping.option_availability_code',
                '=',
                KeyloopFleetKompactOptionAvailabilityType::OPTIONAL
            )
            ->where('keyloop_lease_variant_option_mapping.option_type', KeyloopFleetKompactOptionType::COLOUR);
    }

    public function residualValues(): HasMany
    {
        return $this->hasMany(KeyloopLeaseResidualValue::class, 'external_variant_id', 'external_variant_id');
    }

    public function getDiscounts(string $type = 'all'): Collection
    {
        // do one lookup to get all discounts applicable to this vehicle make
        $make_discounts = Cache::remember(
            tenant('id') . '-lease-make-discounts-' . $this->make_id . '-' . $type,
            Carbon::now()->addMinutes(5),
            fn() => KeyloopLeaseDiscount::query()
                ->where('make_id', '=', $this->make_id)
                ->where('discount_amount', '>', 0)
                ->when($type == 'percentage', fn ($query) => $query->where('discount_type', 'percentage'))
                ->when($type == 'cash', fn ($query) => $query->where('discount_type', 'cash'))
                ->get()
        );

        // filter discounts to get discounts for this model only
        $model_discounts = $make_discounts->filter(
            fn($discount) => $discount->model_id == $this->model_id && empty($discount->external_variant_id)
        );

        // filter discounts to get discounts for this variant only
        $variant_discounts = $make_discounts->filter(
            fn($discount) => $discount->external_variant_id == $this->external_variant_id
        );

        // filter discounts to get fallback discounts for this make, where no model or variant ID is set
        $make_discounts = $make_discounts->filter(
            fn($discount) => empty($discount->model_id) && empty($discount->external_variant_id)
        );

        if (count($variant_discounts)) {
            return $variant_discounts;
        } elseif (count($model_discounts)) {
            return $model_discounts;
        } elseif (count($make_discounts)) {
            return $make_discounts;
        }

        return collect();
    }

    public function scopeSetSelections(Builder $query, array $selections = [])
    {
        collect($selections)
            ->each(function ($selection) use ($query) {
                match ($selection['type']) {
                    'type',
                    'make_id',
                    'model_id',
                    'fuel_type_id',
                    'body_style_id',
                    'transmission_id' => $query->when(
                        in_array(null, $selection['values']),
                        fn($subQuery) => $subQuery->where(fn($nullableQuery) => $nullableQuery
                            ->whereNull($selection['type'])
                            ->orWhereIn($selection['type'], $selection['values'])),
                        fn($subQuery) => $subQuery->whereIn($selection['type'], $selection['values'])
                    ),
                    default => $query
                };
            });
    }

    public function labels(): BelongsToMany
    {
        return $this->belongsToMany(Label::class, 'lease_vehicle_labels', 'vehicle_id');
    }

    public function scopeActive(Builder $query): void
    {
        $query
            ->where('price_ex_vat_ex_vrt', '>', 0)
            ->where('cheapest_monthly_price_ex_vat', '>', 0);
    }

    public function scopeWithRelationshipsForCardView(Builder $query): void
    {
        $query->with([
            'make',
            'model',
            'fuelType',
            'transmission',
            'bodyStyle'
        ]);
    }

    /**
     * We expect variant names to optionally contain duplicate make and model
     * e.g. Audi A3 A3 TFSIE 204HP SLINE
     * This function attempts to remove unnecessary text.
     *
     * @return mixed
     */
    public function getDerivativeAttribute()
    {
        $derivative = $this->name;
        $make_name = $this->make?->name;
        $model_name = $this->model?->name;

        // if make or model is empty, return the unaltered derivative
        if (empty($make_name) || empty($model_name)) {
            return $derivative;
        }

        $model_plus_model = $model_name . ' ' . $model_name;

        // if derivative starts with 'Audi', remove 'Audi'
        if (Str::startsWith($derivative, $make_name)) {
            $derivative = trim(Str::replaceFirst($make_name, '', $derivative));
        }

        // if derivative starts with 'A3 A3', remove 'A3 A3'
        if (Str::startsWith($derivative, $model_plus_model)) {
            $derivative = trim(Str::replaceFirst($model_plus_model, '', $derivative));
        }

        // if derivative starts with 'A3', remove 'A3'
        if (Str::startsWith($derivative, $model_name)) {
            $derivative = trim(Str::replaceFirst($model_name, '', $derivative));
        }

        return trim($derivative);
    }
}
