<?php

namespace Mtc\Money;

use Illuminate\Support\Facades\Event;
use Mtc\Money\Events\LoadPriceModifiers;

/**
 * Trait HasPrices
 *
 * Implements automated and efficient way to automatically get price objects for attributes
 *
 * @package Mtc\Money
 */
trait HasPrices
{

    /**
     * Price objects on model
     *
     * @var Price[]
     */
    private $price_models = [];

    /**
     * Whether to load ex vat price instead of inc vat price
     *
     * @var bool
     */
    protected $load_ex_vat_price = false;


    /**
     * Get an attribute from the model.
     *
     * @param  string $key
     * @return mixed
     */
    public function getAttribute($key)
    {
        if ($this->isPriceAttribute($key)) {
            return $this->fetchPriceAttribute($key);
        }

        return parent::getAttribute($key);
    }

    /**
     * Check if the attribute key is part of the price attribute set
     *
     * @param $key
     * @return bool
     */
    public function isPriceAttribute($key): bool
    {
        if (empty($this->price_fields) || !array_key_exists($key, $this->attributes)) {
            return false;
        }

        return (is_array($this->price_fields) && in_array($key, $this->price_fields));
    }

    /**
     * Get the price value for an attribute
     *
     * @param $key
     * @return mixed
     * @throws \Exception
     */
    public function fetchPriceAttribute($key)
    {
        if (!empty($this->price_models[$key])) {
            return $this->price_models[$key];
        }

        return $this->loadPriceModel($key);
    }

    /**
     * Set up price object on a model attribute
     *
     * @param $key
     * @return mixed
     * @throws \Exception
     */
    public function loadPriceModel($key)
    {
        if ($this->exists == false) {
            $this->attributes[$key] = 0;
        }

        $config = [];
        if (!empty($this->load_ex_vat_price) && is_string(array_search($key, $this->price_fields))) {
            $fetch_key = array_search($key, $this->price_fields);
            $config['price_entered_with_tax'] = false;
        } else {
            $fetch_key = $key;
        }

        $rounded_price_attribute = round($this->attributes[$fetch_key], config('currencies.round.precision'), config('currencies.round.mode'));
        $price = new Price($rounded_price_attribute, $this->taxAttribute($key), $config);
        $price->setModifiers(Event::dispatch(new LoadPriceModifiers($this, $key, $price->tax_rate)));
        $price->calculate();
        $this->price_models[$key] = $price;

        return $this->price_models[$key];
    }


    /**
     * Set up price object on a model attribute based on a given value
     *
     * @param $value
     * @param $key
     * @return mixed
     * @throws \Exception
     */
    public function loadPriceFromValue($value, $key)
    {
        $rounded_price_attribute = round($value, config('currencies.round.precision'), config('currencies.round.mode'));
        $price = new Price($rounded_price_attribute, $this->taxAttribute($key));
        $price->setModifiers(Event::dispatch(new LoadPriceModifiers($this, $key, $price->tax_rate)));
        $price->calculate();
        $this->price_models[$key] = $price;

        return $this->price_models[$key];
    }

    /**
     * Find the tax attribute
     *
     * @param $key
     * @return null
     */
    public function taxAttribute($key)
    {
        if (empty($this->tax_attribute)) {
            return null;
        }

        // Single vat attribute for model
        if (is_string($this->tax_attribute)) {
            return $this->atttributes[$this->tax_attribute] ?? null;
        }

        // Vat set per fields
        if (is_array($this->tax_attribute) && array_key_exists($key, $this->tax_attribute)) {
            return $this->attributes[$this->tax_attribute[$key]];
        }

        return null;
    }
}
