<?php

namespace Mtc\Money;

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Collection;

/**
 * Price Class
 *
 * Class that manages price display on site.
 * Offers internal methods for dealing with inc/ex vat.
 *
 * @package Mtc\Money
 * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Price implements Arrayable
{
    /**
     * The original price, without any modifiers
     *
     * @var float
     */
    protected $original_price = 0;

    /**
     * The default price, without any tax if applicable
     *
     * @var float
     */
    protected $price_excluding_tax = 0;

    /**
     * The default price, including tax if applicable
     *
     * @var float
     */
    protected $price_including_tax = 0;
    /**
     * The default price, including tax if applicable
     *
     * @var float
     */
    protected $tax_on_price = 0;

    /**
     * The config array set within config/tax
     *
     * @var array
     */
    protected $config = [];

    /**
     * The currently set tax rate, and it's applicable rates.
     *
     * @var float
     */
    public $tax_rate = 0;

    /**
     * The currently set tax rate as text value
     *
     * @var string
     */
    public $tax_rate_name = '';

    /**
     * List of all modifiers that have been applied to a price
     *
     * @var Collection
     */
    protected $modifiers;

    /**
     * Build a new Price instance, setting the defined attributes.
     *
     * @param float $price Price numeric value
     * @param string|null $tax_rate_name Key of rate from tax.rates
     * @param array $config
     */
    public function __construct($price, $tax_rate_name = null, $config = [])
    {
        $this->tax_rate_name = $tax_rate_name;
        $this->modifiers = collect([]);
        $this->original_price = $price;
        $this->loadConfig($config);
    }

    /**
     * Load the price config
     *
     * @param $tax_rate_name
     */
    protected function loadConfig($config): void
    {
        // Get config for tax
        $this->config = array_merge(config('tax', []), $config);


        if ($this->tax_rate_name === null) {
            $this->tax_rate_name = $this->config['default_vat_rate'];
        }

        // Get the rates for the specified key.
        if (is_string($this->tax_rate_name) && array_key_exists($this->tax_rate_name, $this->config['vat_rates'])) {
            $this->tax_rate = $this->config['vat_rates'][$this->tax_rate_name];
        } elseif (in_array($this->tax_rate_name, $this->config['vat_rates'])) {
            $this->tax_rate = $this->tax_rate_name;
        }
    }

    /**
     * Calculate values for price object
     */
    public function calculate()
    {
        $price = $this->original_price + $this->modifiers->sum('value');

        if ($this->config['price_entered_with_tax']) {
            $this->calculateEnteredWithTax($price);
        } else {
            $this->calculateEnteredWithoutTax($price);
        }
    }

    /**
     * Calculate values for price object when Price is entered with Tax
     *
     * @param  float $price Price Value
     */
    protected function calculateEnteredWithTax($price): void
    {
        if ($this->checkTaxApplicable()) {
            $this->tax_on_price = round($price - ($price / (1 + $this->tax_rate)), $this->config['price_calculation_precision']);
        }

        $this->price_excluding_tax = $price - $this->tax_on_price;
        $this->price_including_tax = $price;
    }

    /**
     * Calculate values for price object
     *
     * @param  float $price Price Value
     */
    protected function calculateEnteredWithoutTax($price): void
    {
        if ($this->checkTaxApplicable()) {
            $this->tax_on_price = round($price * $this->tax_rate, $this->config['price_calculation_precision']);
        }

        $this->price_excluding_tax = $price;
        $this->price_including_tax = $price + $this->tax_on_price;
    }

    /**
     * Check if Tax is applicable on price
     *
     * @return bool
     */
    protected function checkTaxApplicable(): bool
    {
        return  $this->config['enabled'] === true;
    }

    /**
     * Convert price object to string
     *
     * @return string
     */
    public function __toString()
    {
        $formatter = new \NumberFormatter('en_GB', \NumberFormatter::CURRENCY);
        $currency = Facades\Currency::getCurrentCurrency();

        if ($this->config['display_with_tax']) {
            return $formatter->formatCurrency($this->price_excluding_tax, $currency) . $this->config['price_display_suffix'];
        }

        return $formatter->formatCurrency($this->price_including_tax, $currency) . $this->config['price_display_suffix'];
    }

    /**
     * Get the raw value of the price in display format
     *
     * @return float
     */
    public function original(): float
    {
        return $this->original_price ?? 0;
    }

    /**
     * Get the raw value of the price in display format
     *
     * @param bool|null $display_with_tax
     * @return float
     */
    public function raw($display_with_tax = null, $precision = null): float
    {
        $display_with_tax = $display_with_tax ?? $this->config['display_with_tax'];
        $precision = $precision ?? $this->config['price_calculation_precision'];
        return $display_with_tax ? round($this->price_including_tax, $precision) : round($this->price_excluding_tax, $precision);
    }


    /**
     * Get the raw value of the price in display format
     *
     * @return float
     */
    public function tax($precision = 2): float
    {
        return round($this->tax_on_price, $precision, PHP_ROUND_HALF_DOWN);
    }

    /**
     * Get the tax value as formatted price
     *
     * @return float
     */
    public function taxPrice(): string
    {
        return money_format('%.2n', $this->tax_on_price);
    }

    /**
     * Get the Tax rate name
     *
     * @return string
     */
    public function taxRate(): string
    {
        return $this->tax_rate_name;
    }

    /**
     * Get the raw value of the price in display format
     *
     * @return float
     */
    public function withTax(): float
    {
        return $this->price_including_tax;
    }

    /**
     * Get the raw value of the price in display format
     *
     * @return float
     */
    public function withoutTax(): float
    {
        return $this->price_excluding_tax;
    }

    /**
     * Set modifiers on price
     *
     * @param $modifier_list
     */
    public function setModifiers($modifiers)
    {
        $this->modifiers = collect($modifiers)
            ->filter(function ($modifier) {
                return $modifier instanceof PriceModifier;
            })->sortBy(function ($modifier) {
                return $modifier->order;
            });
    }

    /**
     * Fetch modifiers on price
     *
     * @return Collection
     */
    public function getModifiers()
    {
        return $this->modifiers;
    }

    /**
     * Multiply price by quantity
     *
     * @param int $quantity
     */
    public function multiply(int $quantity = 1)
    {
        $this->price_including_tax *= $quantity;
        $this->price_excluding_tax *= $quantity;
    }


    /**
     * Get the instance as an array.
     *
     * @return array
     */
    public function toArray()
    {
        return [
            'tax' => number_format($this->tax(), 2, '.', ''),
            'original' => number_format($this->original(), 2, '.', ''),
            'with_tax' => number_format($this->withTax(), 2, '.', ''),
            'without_tax' => number_format($this->withoutTax(), 2, '.', ''),
        ];
    }
}
