<?php

namespace Mtc\MotorCheck\Services;

use Mtc\AutomotiveSpecs\VehicleRisks;
use Mtc\AutomotiveSpecs\VehicleValuation;
use Mtc\MotorCheck\Contracts\MotorCheckRiskMapContract;
use Mtc\MotorCheck\Contracts\MotorSpecsToValuationMapContract;
use Mtc\MotorCheck\Exceptions\NotFoundException;
use Mtc\AutomotiveSpecs\Contracts\CarValuationService;
use Mtc\AutomotiveSpecs\Contracts\RiskValuationService;
use Mtc\AutomotiveSpecs\Contracts\SpecSyncService;
use Mtc\AutomotiveSpecs\VehicleSpecification;
use Mtc\MotorCheck\Contracts\MotorSpecsToVehicleSpecificationContract;
use Mtc\MotorCheck\Traits\CachesRisks;
use Mtc\MotorCheck\Traits\CachesSpecs;
use Mtc\MotorCheck\Traits\CachesValuations;

class MotorCheckService implements SpecSyncService, CarValuationService, RiskValuationService
{
    use CachesSpecs;
    use CachesValuations;
    use CachesRisks;

    public function __construct(
        protected array $config,
        protected MotorSpecsApi $specs_api,
        protected MotorCheckApi $check_api,
        protected MotorSpecsToVehicleSpecificationContract $spec_map,
        protected MotorSpecsToValuationMapContract $valuation_map,
        protected MotorCheckRiskMapContract $risk_map,
    ) {
        //
    }

    public function getCarSpecs(string $vrm, int $mileage = 0): VehicleSpecification
    {
        if (empty(trim($vrm))) {
            throw new NotFoundException('VRM missing');
        }

        if ($this->canUseCachedSpecsResponse($vrm, $mileage)) {
            $spec_data = $this->loadSpecsFromCache($vrm);
        } else {
            $spec_data = $this->specs_api->getSpecs($vrm, $mileage);
            $this->cacheSpecsIfAble($vrm, $mileage, $spec_data);
        }
        return $this->spec_map->map($spec_data);
    }

    public function getValuation(string $vrm, int $mileage = 0): VehicleValuation
    {
        if (empty(trim($vrm))) {
            throw new NotFoundException('VRM missing');
        }

        if ($this->canUseCachedValuationResponse($vrm)) {
            $valuation_data = $this->loadValuationFromCache($vrm);
        } else {
            switch ($this->config['valuation_driver']) {
                case 'vip':
                    $valuation_data = $this->specs_api->getVipValuation($vrm, $mileage);
                    break;
                default:
                case 'auto-trader':
                    $valuation_data = $this->specs_api->getAutoTraderValuation($vrm, $mileage);
                    break;
            }
            $this->cacheValuationIfAble($vrm, $mileage, $valuation_data);
        }
        return $this->valuation_map->map($valuation_data);
    }

    public function getCarRisks(string $vrm): VehicleRisks
    {
        if (empty(trim($vrm))) {
            throw new NotFoundException('VRM missing');
        }

        if ($this->canUseCachedRiskResponse($vrm)) {
            $risk_data = $this->loadRisksFromCache($vrm);
        } else {
            $risk_data = $this->check_api->getReport($vrm);
            $this->cacheRisksIfAble($vrm, $risk_data);
        }

        return $this->risk_map->map($risk_data);
    }
}
