# MotorCheck / MotorSpecs API 

This package works with MotorCheck / MotorSpecs API data retrieval and formatting
to a uniform style for further implementation.

This module allows:

* retrieving key specification information on vehicles
* retrieving vehicle valuation through for retail / part exchange
* perform trade checks on a vehicle

# Installation and Setup

Require package:
```shell
composer require mtcmedia/motor-check-data
```

Migrate local cache tables
```shell
php artisan migrate
```

Set .env values for API accounts. 
```dotenv
MOTOR_CHECK_USERNAME=
MOTOR_CHECK_PASSWORD=password
MOTOR_SPEC_CLIENT_ID=client-id
MOTOR_SPEC_SECRET=secret
MOTOR_SPEC_VALUATION_DRIVER=auto-trader #or vip
MOTOR_SPEC_LIVE=true
```
MotorCheck has 2 API domains - MotorSpec and
MotorCheck.

* MotorCheck performs vehicle reports (risk checks) and has only live
environment. Account setup is done from https://trade.motorcheck.co.uk/
and api details are set up at https://trade.motorcheck.co.uk/app/admin/api-credentials
* MotorSpec does have a test domain so `MOTOR_SPEC_LIVE` key is used to specify
whether test or live API domain is used. Account is created by sending a contact
request for creating account form the motorcheck website.
* MotorSpec offers 2 valuation services - VIP and AutoTrader. VIP is cheaper but
not as reliable as AutoTrader. Chosen service can be specified via `MOTOR_SPEC_VALUATION_DRIVER` key. 

To set MotorCheck for `AutoSpecs` facade provided in `mtcmedia/automotive-specs` 
you will need to set the .env value `AUTOMOTIVE_SPECS_DRIVER=motor-check`

(Optional) if you want to change aspects like how data is mapped or whether 
API results are cached locally you can publish the config

```shell
php artisan vendor:publish --provider="Mtc\MotorCheck\Providers\MotorCheckProvider"
```

# Usage

This package is aimed to be used together with its dependency 
`mtcmedia/automotive-specs` by using `AutoSpecs` facade but independent
calls using the service class are available. 


Get Specs / Valuation / Risks
```php
use Mtc\AutomotiveSpecs\Facades\AutoSpecs;
$vrm = 'ABC123';
$mileage = 100000;
$specs = AutoSpecs::getCarSpecs($vrm, $mileage); 
$valuation = AutoSpecs::getValuation($vrm, $mileage); 
$risks = AutoSpecs::getCarRisks($vrm);
```

Alternatively these actions can be called directly on service class

```php
use Mtc\MotorCheck\Services\MotorCheckService;
use Illuminate\Support\Facades\App;
$vrm = 'ABC123';
$mileage = 100000;
$motor_check = App::make(MotorCheckService::class);
$specs = $motor_check->getCarSpecs($vrm, $mileage); 
$valuation = $motor_check->getValuation($vrm, $mileage); 
$risks = $motor_check->getCarRisks($vrm);
```

`getCarSpecs()` method() returns an instance of `\Mtc\AutomotiveSpecs\VehicleSpecification` 

`getValuation()` method returns an instance of `\Mtc\AutomotiveSpecs\VehicleValuation` 

`getCarSpecs()` method returns an instance of `\Mtc\AutomotiveSpecs\VehicleRisks` 


# Contributing

Please see [Contributing](CONTRIBUTING.md) for details.
