<?php

namespace Tests\Unit;

use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\Http;
use Mtc\MotorCheck\Exceptions\NotFoundException;
use Mtc\MotorCheck\Services\MotorCheckApi;
use Mtc\Tests\TestCase;

class MotorCheckApiTest extends TestCase
{
    protected function getEnvironmentSetUp($app)
    {
        $existing_report_fake = [
            'data' => [
                [
                    'report_id' => 'DADEDO'
                ]
            ]
        ];
        $search_sequence = Http::sequence()
            ->push($existing_report_fake, 200)
            ->push([], 200)
            ->whenEmpty(Http::response());

        Http::fake([
            "https://trade-api.motorcheck.co.uk/api/v1/oauth/token" =>  Http::response(['access_token' => 'mc-token']),
            "https://trade-api.motorcheck.co.uk/api/v1/report/show" =>  Http::response(['existing' => 'report']),
            "https://trade-api.motorcheck.co.uk/api/v1/report/create" =>  Http::response(['create' => 'new-report']),
            "https://trade-api.motorcheck.co.uk/api/v1/report/search" =>  $search_sequence,
            '*' => Http::response(['result' => 'wildcard'])
        ]);
    }

    protected function config()
    {
        return [
            'username' => config('motor_check.username') ,
            'password' => config('motor_check.password'),
        ];
    }

    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testGetExistingReport()
    {
        $result = (new MotorCheckApi($this->config()))->getReport('ABC123');

        $expected_result = ['existing' => 'report'];
        self::assertEquals($expected_result, $result);

        Http::assertSent(function (Request $request) {
            return $request->hasHeader('Content-Type', 'application/json')
                && $request->url() === 'https://trade-api.motorcheck.co.uk/api/v1/report/search'
                && $request->body() === json_encode([ 'vrm' => 'ABC123' ]);
        });

        Http::assertSent(function (Request $request) {
            return $request->hasHeader('Content-Type', 'application/json')
                && $request->url() === 'https://trade-api.motorcheck.co.uk/api/v1/report/show'
                && $request->body() === json_encode([ 'report_id' => 'DADEDO' ]);
        });
    }

    public function testSearchOnlyReportNotFound()
    {
        $this->expectException(NotFoundException::class);
        // First sequence - existing
        (new MotorCheckApi($this->config()))->getReport('ABC123', true);
        // second sequence
        (new MotorCheckApi($this->config()))->getReport('ABC123', true);
    }



    public function testGetNewReport()
    {
        // First sequence - existing
        (new MotorCheckApi($this->config()))->getReport('ABC123');
        $result = (new MotorCheckApi($this->config()))->getReport('IL894A');
        $expected = [
            'create' => 'new-report',
        ];

        self::assertEquals($expected, $result);
        Http::assertSent(function (Request $request) {
            return $request->hasHeader('Content-Type', 'application/json')
                && $request->url() === 'https://trade-api.motorcheck.co.uk/api/v1/report/search'
                && $request->body() === json_encode([ 'vrm' => 'IL894A' ]);
        });
        Http::assertSent(function (Request $request) {
            return $request->hasHeader('Content-Type', 'application/json')
                && $request->url() === 'https://trade-api.motorcheck.co.uk/api/v1/report/create'
                && $request->body() === json_encode([
                    'vrm' => 'IL894A',
                    'method' => 'tradeCheck'
                ]);
        });
    }

}
