<?php

namespace Tests\Unit;

use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Mtc\MotorCheck\Exceptions\NotFoundException;
use Mtc\MotorCheck\Exceptions\UnauthorizedRequestException;
use Mtc\MotorCheck\Services\MotorSpecsApi;
use Mtc\Tests\TestCase;

class MotorSpecsApiTest extends TestCase
{
    protected function getEnvironmentSetUp($app)
    {
        Http::fake([
            "https://staging.motorspecs.com/oauth" =>  Http::response(['access_token' => 'test-token']),
            "https://staging.motorspecs.com/identity-specs/lookup" =>  Http::response(['identity-specs' => 'test']),
            "https://staging.motorspecs.com/specs/standard" =>  Http::response(['standard-features' => 'value-response']),
            "https://staging.motorspecs.com/valuation-autotrader/value" =>  Http::response(['auto-trader' => 'valuation']),
            "https://staging.motorspecs.com/valuation-vip/value" =>  Http::response(['vip-valuation' => 'high-value-item']),
            "https://api.motorspecs.com/oauth" =>  Http::response(['access_token' => 'production-token']),
            "https://api.motorspecs.com/valuation-vip/value" =>  Http::response(['vip-valuation' => 'production-value']),
            '*' => Http::response(['result' => 'wildcard'])
        ]);
    }

    protected function config($live = false)
    {
        return [
            'live' => $live,
            'username' => config('motor_check.specs_username'),
            'password' => config('motor_check.specs_password'),
            'country' => 'gb'
        ];
    }

    public function testSendingSpecsRequestHeadersAndPayload()
    {
        (new MotorSpecsApi($this->config()))->getSpecs('ABC123', 123);

        $specs_lookup_params = [
            'registration' => 'ABC123',
            'currentMiles' => 123,
            'country' => 'gb'
        ];

        Http::assertSent(function (Request $request) use ($specs_lookup_params) {
            return $request->hasHeader('Content-Type', 'application/vnd.identity-specs.v2+json')
                && $request->url() === 'https://staging.motorspecs.com/identity-specs/lookup'
                && $request->body() === json_encode($specs_lookup_params);
        });

        $standard_lookup_params = [
            'registration' => 'ABC123',
            'country' => 'gb'
        ];

        Http::assertSent(function (Request $request) use ($standard_lookup_params) {
            return $request->hasHeader('Content-Type', 'application/vnd.specs.v2+json')
                && $request->url() === 'https://staging.motorspecs.com/specs/standard'
                && $request->body() === json_encode($standard_lookup_params);
        });
    }

    public function testSpecsRequestResponseReturnedOnApiCall()
    {
        $result = (new MotorSpecsApi($this->config()))->getSpecs('ABC123', 123);

        $expected = [
            'identity' => [
                'identity-specs' => 'test',
            ],
            'standard' => [
                'standard-features' => 'value-response'
            ]
        ];

        self::assertEquals($expected, $result);
    }

    public function testAutoTraderApiHeaders()
    {
        (new MotorSpecsApi($this->config()))->getAutoTraderValuation('ABC123', 123);

        $specs_lookup_params = [
            'registration' => 'ABC123',
            'currentMiles' => 123,
            'country' => 'gb'
        ];

        Http::assertSent(function (Request $request) use ($specs_lookup_params) {
            return $request->hasHeader('Content-Type', 'application/vnd.valuation-auto-trader.v2+json')
                && $request->url() === 'https://staging.motorspecs.com/valuation-autotrader/value'
                && $request->body() === json_encode($specs_lookup_params);
        });

    }

    public function testAutoTraderApiResponseFormat()
    {
        $response = (new MotorSpecsApi($this->config()))->getAutoTraderValuation('ABC123', 123);

        $expected = [
            'auto-trader' => 'valuation'
        ];

        self::assertEquals($expected, $response);
    }

    public function testVipValuationApiHeaders()
    {
        (new MotorSpecsApi($this->config()))->getVipValuation('ABC123', 123);

        $specs_lookup_params = [
            'registration' => 'ABC123',
            'currentMiles' => 123,
            'country' => 'gb'
        ];

        Http::assertSent(function (Request $request) use ($specs_lookup_params) {
            return $request->hasHeader('Content-Type', 'application/vnd.valuation-vip.v2+json')
                && $request->url() === 'https://staging.motorspecs.com/valuation-vip/value'
                && $request->body() === json_encode($specs_lookup_params);
        });
    }

    public function testVipValuationApiResponseFormat()
    {
        $response = (new MotorSpecsApi($this->config()))->getVipValuation('ABC123', 123);

        $expected = [
            'vip-valuation' => 'high-value-item'
        ];

        self::assertEquals($expected, $response);
    }

    public function testLiveEndpointRequest()
    {
        $this->app['config']->set('motor_check.live', true);

        (new MotorSpecsApi($this->config(true)))->getVipValuation('ABC123', 123);

        Http::assertSent(function (Request $request) {
            return  $request->url() === 'https://api.motorspecs.com/valuation-vip/value';
        });
    }
}
