<?php

namespace Mtc\MotorCheck\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Mtc\MotorCheck\Exceptions\NotFoundException;
use Mtc\MotorCheck\Exceptions\UnauthorizedRequestException;

class MotorCheckApi
{
    public function __construct(protected array $config = [])
    {
        //
    }

    /**
     * Get a report from MotorCheck.
     * Shows risk details (stolen, finance in place etc) based on VRM.
     *
     * @param string $vrm
     * @param bool $search_only
     * @return array
     * @throws NotFoundException
     */
    public function getReport(string $vrm, bool $search_only = false)
    {
        $report_id = $this->searchReport($vrm);
        if ($search_only && $report_id === null) {
            throw new NotFoundException('Unable to find report for vehicle');
        }

        return !empty($report_id)
            ? $this->showReport($report_id)
            : $this->createReport($vrm);
    }

    /**
     * Search for an existing report on MC service
     *
     * @param $vrm
     * @return string|null
     * @throws \Illuminate\Http\Client\RequestException
     */
    protected function searchReport($vrm): string|null
    {
        $response = Http::withToken($this->getAccessToken())
            ->withHeaders($this->headers())
            ->post($this->endpoint('api/v1/report/search'), [
                'vrm' => $vrm,
            ]);

        return $response->throw()->json('data.0.report_id');
    }

    /**
     * Create a new report on MC service.
     * This will charge money for usage whilst search/show does not cost money
     *
     * @param $vrm
     * @return array
     * @throws \Illuminate\Http\Client\RequestException
     */
    protected function createReport($vrm): array
    {
        return Http::withToken($this->getAccessToken())
            ->withHeaders($this->headers())
            ->post($this->endpoint('api/v1/report/create'), [
                'vrm' => $vrm,
                'method' => 'tradeCheck',
            ])
            ->throw()
            ->json();
    }

    /**
     * Show an existing MC report based on report ID
     *
     * @param $report_id
     * @return array
     * @throws \Illuminate\Http\Client\RequestException
     */
    protected function showReport($report_id): array
    {
        $response = Http::withToken($this->getAccessToken())
            ->withHeaders($this->headers())
            ->post($this->endpoint('api/v1/report/show'), [
                'report_id' => $report_id,
            ]);

        return $response->throw()->json();
    }

    protected function getAccessToken(): string
    {
        return Cache::remember('motor-check-api-token', now()->addDay(), function () {
            return $this->login();
        });
    }

    protected function login(): string
    {
        $result = Http::post($this->endpoint('api/v1/oauth/token'), [
                'scope' => 'report-search report-show report-create',
                'grant_type' => 'client_credentials',
                'client_id' => $this->config['username'],
                'client_secret' => $this->config['password'],
            ]);

        if ($result->failed()) {
            throw new UnauthorizedRequestException('Failed to authorize');
        }

        return $result->json('access_token');
    }

    protected function endpoint(string $path): string
    {
        return 'https://trade-api.motorcheck.co.uk/' . $path;
    }

    protected function headers(): array
    {
        return [
            'Content-Type: application/json',
            'Accept: application/json',
        ];
    }
}
