<?php

namespace Mtc\MtcPay;

use Mtc\MtcPay\Exceptions\ConfigException;
use Mtc\MtcPay\Exceptions\ApiException;

/**
 * Communicate between MTC Pay's API.
 *
 * @package  Mtc\MtcPay
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class ApiClient
{
    /**
     * Store a copy of the API config after it has been decoded.
     * @var array $config
     */
    protected static $config = [];

    public function __construct()
    {
        $this->getConfig();
    }

    /**
     * Decode the 'mtc_pay.config' key to it's provided values
     * @return void
     */
    protected function getConfig(): void
    {
        if (!empty(self::$config)) {
            return;
        }

        if (empty(config('mtc_pay.config'))) {
            throw new ConfigException("mtc_pay.config has not been defined");
        }

        $config = json_decode(base64_decode(config('mtc_pay.config')), true);
        if (empty($config)) {
            throw new ConfigException("mtc_pay.config could not be decoded: " . json_last_error_msg());
        }

        self::$config = $config;
    }

    /**
     * Send an API request to the MTC Pay API.
     *
     * @param string $action API Resource path (e.g. checkout)
     * @param string $method
     * @param array $data
     * @throws ApiException
     * @return array
     */
    public function call($action, $method = 'GET', $data = [])
    {
        $curl = curl_init();

        if ($method === 'GET') {
            $action .= http_build_query($data);
        } else {
            curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($data));
        }

        curl_setopt_array($curl, [
            CURLOPT_URL => self::$config['base_url'] . '/' . $action,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . self::$config['api_key'],
            ],
        ]);

        $response = curl_exec($curl);

        if ($error = curl_error($curl)) {
            throw new ApiException("Issue retrieving response: " . $error);
        }

        $response_data = json_decode($response, true);

        if (empty($response_data) || !is_array($response_data)) {
            throw new ApiException("Issue decoding response: " . json_last_error_msg());
        }

        return $response_data;
    }
}