<?php

namespace Mtc\MultiBuy;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Money\HasPrices;
use Mtc\Money\Price;
use Illuminate\Database\Eloquent\Builder;

/**
 * Class Discount
 * @package Mtc\MultiBuy
 */
class Discount extends Model
{
    use HasPrices;

    /**
     * Sale restriction constants
     */
    public const ALL_ITEMS = 0;
    public const NON_SALE_ITEMS = 1;
    public const ONLY_SALE_ITEMS = 2;

    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'name',
        'type',
        'value1',
        'value2',
        'disabled',
        'order',
        'basic_restrictions',
        'sale_restriction'
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'valid_from' => 'datetime',
        'valid_to' => 'datetime',
        'disabled' => 'boolean',
        'basic_restrictions' => 'array',
        'sale_restriction' => 'integer'
    ];

    /**
     * Extend model booting
     */
    protected static function boot()
    {
        parent::boot();

        self::deleting(function (self $discount) {
            $discount->restrictions()->delete();
        });
    }

    /**
     * Define the relationship to discount restrictions
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function restrictions(): HasMany
    {
        return $this->hasMany(Restriction::class);
    }

    /**
     * Stackables (Discounts which can apply with this discount)
     *
     * @return BelongsToMany
     */
    public function stackables(): BelongsToMany
    {
        return $this->belongsToMany(Discount::class, 'discounts_stackables', 'discount_id', 'stackable_discount_id');
    }

    /**
     * @param Request $request
     * @return Builder
     */
    public function scopeSearch(Builder $query, Request $request): Builder
    {
        collect(config('discounts.admin_search_filters', []))
            ->each(function ($filter_class) use ($request, $query) {
                App::make($filter_class)->handle($request, $query);
            });

        return $query->orderBy('order');
    }

    /**
     * Scope - enabled()
     * Used to get enabled discounts
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeEnabled(Builder $query): Builder
    {
        return $query->where('disabled', 0);
    }

    /**
     * Scope - active()
     * Discards all hidden and deleted items
     *
     * @param Builder $query Query builder object
     * @return Builder
     */
    public function scopeActive($query): Builder
    {
        return $query->enabled()
            ->where(function ($from) {
                return $from->where('valid_from', '<=', Carbon::now())
                    ->orWhereNull('valid_from');
            })
            ->where(function ($to) {
                return $to->where('valid_to', '>=', Carbon::now())
                    ->orWhereNull('valid_to');
            });
    }

    /**
     * Get the name value
     *
     * @return mixed
     */
    public function getNameAttribute()
    {
        return __('discounts::discounts.basket_name', [
            'name' => $this->attributes['name']
        ]);
    }

    /**
     * Get Discount Amount attribute
     *
     * @return mixed
     * @throws \Exception
     */
    public function getAmountAttribute()
    {
        if (empty($this->price_models['amount'])) {
            $config = config('basket.discount_display_with_vat')
                ? []
                : [
                    'price_entered_with_tax' => false
                ];
            $price = new Price($this->applicableToBasket(App::make('basket')), null, $config);
            $price->calculate();
            $this->price_models['amount'] = $price;
        }

        return $this->price_models['amount'];
    }

    /**
     * Check if discount is active
     *
     * @return string
     */
    public function getStatusAttribute()
    {
        if ($this->disabled) {
            return 'Disabled';
        }

        if ($this->valid_to && $this->valid_to < Carbon::now()) {
            return 'Expired ' . $this->valid_to->format('Y-m-d');
        }

        if ($this->valid_from && $this->valid_from >= Carbon::now()) {
            return 'Upcoming';
        }

        return 'Active';
    }

    /**
     * Multi buy discounts are site-wide discounts and are not removable from basket
     *
     * @return bool
     */
    public function getIsRemovableAttribute()
    {
        return false;
    }

    /**
     * Get the applicable amount on basket
     *
     * @param $basket
     * @return mixed
     */
    public function applicableToBasket($basket)
    {
        $this->deductable_amount = $basket->getCostSubtotalAttribute();
        App::make(config('discounts.discount_types.' . $this->type . '.class'))->apply($this, $basket);
        return $this->discounted_amount;
    }

    /**
     * Get Basic Restriction Value
     *
     * @param $restriction
     * @return |null
     */
    public function getBasicRestriction($restriction)
    {
        return isset($this->basic_restrictions[$restriction]) ? $this->basic_restrictions[$restriction] : null;
    }

    /**
     * Get Discounted values
     *
     * @param $basket_item
     * @return mixed
     */
    public function getDiscountedValues($basket_item)
    {
        return App::make(config('discounts.discount_types.' . $this->type . '.class'))
            ->getDiscountedValues($this, $basket_item);
    }
}
