<?php

namespace Mtc\MultiBuy\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\MultiBuy\Contracts\DiscountRepositoryInterface;
use Mtc\MultiBuy\Http\Requests\CreateDiscountRequest;
use Mtc\MultiBuy\Http\Requests\UpdateDiscountRequest;

/**
 * Class DiscountController
 * @package Mtc\MultiBuy\Http\Controllers\Admin
 */
class DiscountController extends Controller
{
    /**
     * @var DiscountRepositoryInterface
     */
    protected $discount_repository;

    /**
     * DiscountController constructor.
     * @param DiscountRepositoryInterface $discount_repository
     */
    public function __construct(DiscountRepositoryInterface $discount_repository)
    {
        $this->discount_repository = $discount_repository;
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        $basic_restrictions = $this->discount_repository->getBasicRestrictions()
            ->reject(function($basic_restriction, $key) {
                return $key === 'minimum_basket_price';
            });

        $this->page_meta['title'] = 'Manage Discounts';
        return template('admin/discounts/index.twig', [
            'page_meta' => $this->page_meta,
            'basic_restrictions' => $basic_restrictions,
            'discounts' => $this->discount_repository->getDiscounts(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Discount';
        return template('admin/discounts/edit.twig', [
            'discount' => $this->discount_repository->getDiscountClass(),
            'page_meta' => $this->page_meta,
            'discount_types' => $this->discount_repository->getTypes(),
            'sale_restrictions' => $this->discount_repository->getSaleRestrictions(),
            'form_action' => route('discounts.store')
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param CreateDiscountRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(CreateDiscountRequest $request) : \Illuminate\Http\RedirectResponse
    {
        $this->discount_repository->updateOrCreateModel($request);
        session()->flash('success', 'Discount Created');
        return redirect()->to(route('discounts.edit', [ $this->discount_repository->getId() ]));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Request $request
     * @param $id
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Request $request, $id)
    {
        $this->discount_repository->findById($id);
        $this->page_meta['title'] = 'Edit Discount';
        return template('admin/discounts/edit.twig', [
            'restrictions' => $this->discount_repository->getRestrictions(),
            'basic_restrictions' => $this->discount_repository->getBasicRestrictions(),
            'sale_restrictions' => $this->discount_repository->getSaleRestrictions(),
            'discount' => $this->discount_repository->getModel(),
            'discount_types' => $this->discount_repository->getTypes(),
            'page_meta' => $this->page_meta,
            'chart' => $this->discount_repository->getChart(),
            'form_action' => route('discounts.update', [ $id ])
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateDiscountRequest $request
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(UpdateDiscountRequest $request, $id) : \Illuminate\Http\RedirectResponse
    {
        $this->discount_repository->updateOrCreateModel($request, $id);
        session()->flash('success', 'Discount Updated');
        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id) : RedirectResponse
    {
        $this->discount_repository->deleteById($id);
        session()->flash('success', 'Discount Deleted');
        return redirect()->back();
    }
}
