<?php

namespace Mtc\MultiBuy\Charts;

use Carbon\Carbon;
use ConsoleTVs\Charts\Classes\Chartjs\Chart;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Mtc\Orders\Contracts\OrderContract;

class Discount extends Chart
{
    /**
     * @var \Mtc\MultiBuy\Discount
     */
    protected $discount;

    /**
     * @var Request
     */
    protected $request;

    /**
     * @var Carbon
     */
    protected $start_date;

    /**
     * @var Carbon
     */
    protected $end_date;

    /**
     * Initializes the chart.
     *
     * @return void
     */
    public function __construct(\Mtc\MultiBuy\Discount $discount, Request $request)
    {
        parent::__construct();

        $this->discount = $discount;
        $this->request = $request;

        $this->processRequest();

        $this->loadOptions();
        $this->loadLabels();
        $this->loadOrdersData();
    }

    /**
     * Processt request
     */
    protected function processRequest()
    {
        $this->start_date = $this->request->has('chart_start_date')
            ? Carbon::createFromFormat('d/m/Y', $this->request->input('chart_start_date'))->startOfDay()
            : Carbon::now()->startOfDay()->subMonth(1);
        $this->end_date = $this->request->has('chart_end_date')
            ? Carbon::createFromFormat('d/m/Y', $this->request->input('chart_end_date'))
            : Carbon::now();
    }

    /**
     * Load chart options
     */
    protected function loadOptions()
    {
        $this->options([
            'scales' => [
                'yAxes' => [
                    [
                        'ticks' => [
                            'stepSize' => config('discounts.chart.step_size', 1)
                        ],
                    ],
                ],
            ],
        ]);
    }

    /**
     * Load labels for chart
     */
    protected function loadLabels()
    {
        $labels = [];
        $start_date = Carbon::parse($this->start_date);

        if ($this->request->input('chart_group_by') === 1) {
            $start_date = $start_date->startOfMonth();
        } elseif ($this->request->input('chart_group_by') === 2) {
            $start_date = $start_date->startOfYear();
        }

        while ($start_date <= $this->end_date) {
            $labels[] = $this->getLabelName($start_date);
            $start_date = $this->updateStartDate($start_date);
        }
        $this->labels($labels);
    }

    /**
     * @param Carbon $date
     * @return int|string
     */
    protected function getLabelName(Carbon $date)
    {
        if ($this->request->input('chart_group_by') === '1') {
            //return $date->format('F');
            return $date->month;
        }

        if ($this->request->input('chart_group_by') === '2') {
            //return $date->format('F');
            return $date->year;
        }

        return $date->toDateString();
    }

    /**
     * @param Carbon $date
     * @return Carbon
     */
    protected function updateStartDate(Carbon $date): Carbon
    {
        if ($this->request->input('chart_group_by') === '1') {
            return $date->addMonth(1);
        }
        if ($this->request->input('chart_group_by') === '2') {
            return $date->addYear(1);
        }

        return $date->addDay(1);
    }

    /**
     * Load orders data
     */
    protected function loadOrdersData()
    {
        $discounts_orders_usage = App::make(OrderContract::class)
            ->where('paid_at', '>=', $this->start_date)
            ->where('paid_at', '<=', $this->end_date)
            ->whereHas('discounts', function ($query) {
                $query->where('discount_type', 'discount')
                    ->where('discount_id', $this->discount->id);
            })
            ->select(['*', DB::raw($this->getDataSelect())])
            ->get()
            ->groupBy('date')
            ->map(function ($discounts) {
                return $discounts->count();
            });

        $data_orders = [];
        foreach ($this->labels as $label) {
            $data_orders[] = $discounts_orders_usage[$label] ?? 0;
        }

        $this->dataset('Orders', 'line', $data_orders);
    }

    /**
     * @return string
     */
    protected function getDataSelect(): string
    {
        if ($this->request->input('chart_group_by') === '1') {
            return 'MONTH(paid_at) as date';
        }

        if ($this->request->input('chart_group_by') === '2') {
            return 'YEAR(paid_at) as date';
        }

        return 'DATE(paid_at) as date';
    }
}
