<?php

namespace Mtc\PayOnCollection;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;

/**
 * Pay On Collection Payment Gateway
 *
 * @package  Mtc\PayOnCollection
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class PayOnCollection implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        return App::make(config('pay_on_collection.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $invoice = template('pay_on_collection/payment_form.twig', [
            'invoice' => $invoice
        ]);
        return new PaymentForm($invoice, 'template', [
            'name' => __('pay_on_collection::pay_on_collection.payment_option_name')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        return [
            'provider' => 'pay-on-collection',
            'amount' => $invoice->getOutstandingAmount(),
            'reference' => config('pay_on_account.reference_prefix') . $invoice->getId(),
            'details' => [],
            'confirmation_status' => config('pay_on_collection.confirmation_status'),
            'confirmed_at' => config('pay_on_collection.manually_confirm_payment_amount') ? null : now(),
        ];

    }

}
