<?php

namespace Mtc\Opayo;

use Carbon\Carbon;
use GuzzleHttp\Client;
use GuzzleHttp\RequestOptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;

class Opayo implements PaymentGateway
{
    /**
     * @var Client
     */
    protected $api;

    /**
     * Opayo constructor.
     * @param Client $client
     */
    public function __construct(Client $client)
    {
        $this->api = $client;
    }

    /**
     * @param InvoiceRepositoryContract $invoice
     * @param \Mtc\Checkout\Contracts\PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('opayo.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        return new PaymentForm('opayo-payment', 'vue-component', [
            'invoice_id' => $invoice->getId(),
            'name' => __('opayo::opayo.payment_option_name'),
            'session_key' => $this->createSessionKey(),
            'test_mode' => Config::get('opayo.test_mode'),
        ]);
    }

    /**
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        // Not used
        return [];

    }

    /**
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array|string[]
     */
    public function attempt(Request $request, InvoiceRepositoryContract $invoice): array
    {
        try {
            $recipient = $invoice->getModel()->recipient;
            if (Config::get('opayo.test_mode')) {
                $recipient->address1 = '88';
                $recipient->postcode = '412';
                $reference_suffix = Carbon::now()->format('U');
            }
            $response = $this->api->post($this->getApiDomain() . 'transactions', [
                RequestOptions::AUTH => [
                    Config::get('opayo.integration_key'),
                    Config::get('opayo.integration_password'),
                ],
                RequestOptions::JSON => [
                    "transactionType" => "Payment",
                    "paymentMethod" => [
                        "card" => [
                            "merchantSessionKey" => $request->input('session_id'),
                            "cardIdentifier" => $request->input('card-identifier'),
                        ]
                    ],
                    "vendorTxCode" => $invoice->getReference() . ($reference_suffix ?? ''),
                    "amount" => (int)($invoice->getOutstandingAmount() * 100),
                    "currency" => $invoice->getCurrency(),
                    "description" => 'Payment on ' . config('app.name') . ' order ' . $invoice->getReference(),
                    "apply3DSecure" => "UseMSPSetting",
                    "customerFirstName" => $recipient->first_name,
                    "customerLastName" => $recipient->last_name,
                    "billingAddress" => [
                        "address1" => $recipient->address1,
                        "city" => $recipient->city ?? '',
                        "postalCode" => $recipient->postcode ?? '',
                        "country" => $recipient->country
                    ],
                    "entryMethod" => Config::get('opayo.transaction_entry_method'),
                ]
            ]);
            $transaction_details = json_decode((string)$response->getBody());
            if ($transaction_details->status === 'Ok') {
                return [
                    'status' => 'ok',
                    'provider' => 'opayo',
                    'amount' => $invoice->getOutstandingAmount(),
                    'currency_code' => $invoice->getCurrency(),
                    'amount_in_currency' => $invoice->getOutstandingAmountInCurrency(),
                    'reference' => $transaction_details->transactionId,
                    'details' => $transaction_details,
                    'confirmed_at' => now(),
                ];
            }

            if ($transaction_details->status === '3DAuth') {
                return (array)$transaction_details;
            }

        } catch (\Exception $exception) {
        }

        return [
            'status' => 'failed'
        ];
    }

    /**
     * @param Request $request
     * @param $transaction_id
     * @return array|false|void
     */
    public function verify(Request $request, $transaction_id)
    {
        return $request->has('cRes')
            ? $this->verifyChallenge($request, $transaction_id)
            : $this->verifyFallback($request, $transaction_id);
    }

    /**
     * @param Request $request
     * @param $transaction_id
     */
    protected function verifyChallenge(Request $request, $transaction_id)
    {
        $response = $this->api->post($this->getApiDomain() . "transactions/{$transaction_id}/3d-secure-challenge", [
            RequestOptions::AUTH => [
                Config::get('opayo.integration_key'),
                Config::get('opayo.integration_password'),
            ],
            RequestOptions::JSON => [
                'cRes' => $request->input('cres')
            ]
        ]);

        $transaction_details = json_decode((string)$response->getBody());
        // TODO: test this
        dd($transaction_details);

    }

    /**
     * @param Request $request
     * @param $transaction_id
     * @return array|false
     */
    protected function verifyFallback(Request $request, $transaction_id)
    {
        $response = $this->api->post($this->getApiDomain() . "transactions/{$transaction_id}/3d-secure", [
            RequestOptions::AUTH => [
                Config::get('opayo.integration_key'),
                Config::get('opayo.integration_password'),
            ],
            RequestOptions::JSON => [
                'paRes' => $request->input('PaRes')
            ]
        ]);

        $verification_details = json_decode((string)$response->getBody());
        if (empty($verification_details->status) || $verification_details->status !== 'Authenticated') {
            return false;
        }

        return $this->getTransaction($transaction_id);
    }

    /**
     * @param $transaction_id
     * @return array|false
     */
    protected function getTransaction($transaction_id)
    {
        $response = $this->api->get($this->getApiDomain() . "transactions/{$transaction_id}", [
            RequestOptions::AUTH => [
                Config::get('opayo.integration_key'),
                Config::get('opayo.integration_password'),
            ],
        ]);

        $transaction_details = json_decode((string)$response->getBody());

        if ($transaction_details->status === 'Ok') {
            return [
                'provider' => 'opayo',
                'amount' => $transaction_details->amount->totalAmount / 100,
                'currency_code' => $transaction_details->currency,
                'amount_in_currency' => $transaction_details->amount->totalAmount / 100,
                'reference' => $transaction_details->transactionId,
                'details' => $transaction_details,
                'confirmed_at' => now(),
            ];
        }
        return false;
    }

    /**
     * @return string|null
     */
    protected function createSessionKey()
    {
        try {
            $response = $this->api->post($this->getApiDomain() . 'merchant-session-keys', [
                RequestOptions::AUTH => [
                    Config::get('opayo.integration_key'),
                    Config::get('opayo.integration_password'),
                ],
                RequestOptions::JSON => [
                    'vendorName' => Config::get('opayo.vendor_name')
                ]
            ]);
            return json_decode((string)$response->getBody())->merchantSessionKey;
        } catch (\Exception $exception) {
            return null;
        }
    }

    /**
     * @return string
     */
    public function getApiDomain()
    {
        return Config::get('opayo.test_mode')
            ? "https://pi-test.sagepay.com/api/v1/"
            : "https://pi-live.sagepay.com/api/v1/";
    }
}