<?php

namespace Mtc\Orders\Factories;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Orders\Contracts\ShipmentFactoryContract;
use Mtc\Orders\Order;
use Mtc\Orders\OrderShipment;


/**
 * Class SplitShipmentFactory
 *
 * Creates split shipments - one shipment for the values set to ship, second for outstanding items
 *
 * @package Mtc\Orders
 */
class SplitShipmentFactory implements ShipmentFactoryContract
{

    /**
     * @var Request
     */
    protected $request;

    /**
     * @var Order
     */
    protected $order;

    /**
     * Create a shipment
     *
     * @param Request $request
     * @param Order $order
     * @return mixed
     */
    public function create(Request $request, OrderContract $order)
    {
        $this->request = $request;
        $this->order = $order;

        $shipment = $this->makeFromRequest();

        $order->refresh();
        if ($order->hasItemsWithoutShipment()) {
            $order->changeStatus(config('orders.statuses.shipped'), Auth::user());
        } else {
            $order->changeStatus(config('orders.statuses.part-shipped'), Auth::user());
            $this->makeOutstanding();
        }

        return $shipment;
    }

    /**
     * Create a shipment
     *
     * @return OrderShipment
     */
    protected function makeFromRequest()
    {
        $shipment_value = $this->calculateValue();
        /** @var OrderShipment $shipment */
        $shipment = $this->order->shipments()
            ->create([
                'shipment_value' => $shipment_value,
                'order_value' => $shipment_value,
                'status' => OrderShipment::STATUS_COMPLETED,
            ]);

        collect($this->request->input('quantity'))
            ->each(function ($quantity, $id) use ($shipment) {
                if ($quantity <= 0) {
                    return;
                }

                $shipment->items()
                    ->create([
                        'order_item_id' => $id,
                        'quantity' => $quantity ?? 0,
                        'ship_weight' => $this->request->input("weight.{$id}", 0) * $quantity,
                    ]);
            });

        return $shipment;
    }

    /**
     * Create a shipment
     *
     */
    protected function makeOutstanding()
    {
        $shipment_value = $this->calculateValue(false);

        /** @var OrderShipment $second_shipment */
        $shipment = $this->order->shipments()
            ->create([
                'shipment_value' => $shipment_value,
                'order_value' => $shipment_value,
                'status' => OrderShipment::STATUS_PENDING,
            ]);

        collect($this->order->items)
            ->each(function ($item) use ($shipment) {
                $quantity = max($item->quantity_not_shipped, 0);
                if ($quantity == 0) {
                    return;
                }

                $shipment->items()
                    ->create([
                        'order_item_id' => $item->id,
                        'quantity' => $quantity,
                        'ship_weight' => $this->request->input("weight.{$item->id}", 0) * $quantity,
                    ]);
            });
    }

    /**
     * Calculate shipment value
     *
     * @param bool $main_shipment
     * @return array|mixed|string|null
     */
    protected function calculateValue($main_shipment = true)
    {
        if (config('order.manual_shipment_value') && $this->request->input('order_value')) {
            return $this->request->input('order_value');
        }

        if ($main_shipment) {
            return collect($this->order->items)
                ->map(function ($item) {
                    return $this->request->input("quantity.{$item->id}", 0) * $item->paid_price->raw();
                })
                ->sum();
        }

        return collect($this->order->items)
            ->map(function ($item) {
                return max($item->quantity_not_shipped, 0) * $item->paid_price->raw();
            })
            ->sum();

    }
}
