<?php

namespace Mtc\Orders;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Event;
use Mtc\Money\Events\LoadPriceModifiers;
use Mtc\Money\HasPrices;
use Mtc\Money\Price;
use Mtc\Shop\Item\Size;

/**
 * Class Item
 *
 * @package Mtc\Orders
 */
class Item extends Model
{
    use SoftDeletes;
    use HasPrices;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'order_items';

    /**
     * Columns that cannot be mass assigned
     *
     * @var array
     */
    protected $guarded = [
        'id'
    ];

    /**
     * When saved touch an update of a relationship
     *
     * @var array
     */
    protected $touches = [
        'order'
    ];

    /**
     * @var array
     */
    protected $appends = [
        'basket_image',
    ];

    /**
     * Cast attributes to types
     *
     * @var array
     */
    protected $casts = [
        'attribute_fields' => 'array',
        'data' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * @var array
     */
    protected $price_fields = [
        'unit_price_ex_vat' => 'unit_price',
        'paid_price_ex_vat' => 'paid_price',
    ];

    protected $tax_attribute = 'vat_rate';

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        self::retrieved(function (self $order_item) {
            if ($order_item->order->use_ex_vat) {
                $order_item->load_ex_vat_price = true;
            }
        });
    }

    /**
     * Relationship with order
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Order Item may have sub-lines to break down details
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function lines()
    {
        return $this->hasMany(Line::class, 'order_item_id');
    }

    /**
     * Relationship with item that is getting purchased
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function purchasable()
    {
        return $this->morphTo();
    }

    /**
     * Items that have already been shipped
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function shipmentItems()
    {
        return $this->hasMany(OrderShipmentItem::class, 'order_item_id');
    }

    /**
     * Relationship with item
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(\Mtc\Shop\Item::class, 'purchasable_id')
            ->where('purchasable_type', \Mtc\Shop\Item::class);
    }

    /**
     * Relationship with item size
     *
     * @return BelongsTo
     */
    public function size(): BelongsTo
    {
        return $this->belongsTo(Size::class, 'purchasable_id')
            ->where('purchasable_type', Size::class);
    }

    /**
     * @return string
     */
    public function getBasketImageAttribute()
    {
        return $this->getBasketImage();
    }

    /**
     * unit_price as Price object
     *
     * @return Price
     */
    public function getOriginalPriceAttribute(): Price
    {
        if ($this->load_ex_vat_price) {
            $config = [
                'price_entered_with_tax' => false
            ];
            $price = new Price($this->attributes['unit_price_ex_vat'] * $this->attributes['quantity'], $this->vat_rate, $config);
        } else {
            $price = new Price($this->attributes['unit_price'] * $this->attributes['quantity'], $this->vat_rate);
        }
        $price->setModifiers(Event::dispatch(new LoadPriceModifiers($this, 'original_price', $price->tax_rate)));
        $price->calculate();
        return $price;
    }

    /**
     * paid_price as Price object
     *
     * @return Price
     */
    public function getLineTotalAttribute(): Price
    {
        if ($this->load_ex_vat_price) {
            $config = [
                'price_entered_with_tax' => false
            ];
            $price = new Price($this->attributes['paid_price_ex_vat'] * $this->attributes['quantity'], $this->vat_rate, $config);
        } else {
            $price = new Price($this->attributes['paid_price'] * $this->attributes['quantity'], $this->vat_rate);
        }
        $price->setModifiers(Event::dispatch(new LoadPriceModifiers($this, 'original_price', $price->tax_rate)));
        $price->calculate();
        return $price;
    }

    /**
     * Filter out variant fields from additional_info data
     *
     * @return array
     */
    public function getVariantFieldsAttribute()
    {
        if ($this->purchasable && !empty($this->purchasable->getVariantFields())) {
            return collect($this->additional_info)
                ->only($this->purchasable->getVariantFields())
                ->toArray();
        }
        return [];
    }

    /**
     * Get the product weight
     *
     * @return int
     */
    public function getWeightAttribute()
    {
        return $this->purchasable ? $this->purchasable->getWeight() : 0;
    }

    /**
     * Get the product stock
     *
     * @return int
     */
    public function getProductStockAttribute()
    {
        return $this->purchasable && $this->purchasable->getStock() > 0 ? $this->purchasable->getStock() : 0;
    }

    /**
     * Find the quantity that is not already assigned to a shipment
     *
     * @return mixed
     */
    public function getShippableQuantityAttribute()
    {
        return max(
            min($this->quantity - $this->shipmentItems->sum('quantity'), $this->getProductStockAttribute()),
            0);
    }

    /**
     * Find the quantity that is not already assigned to a shipment
     *
     * @return mixed
     */
    public function getQuantityNotShippedAttribute()
    {
        return max($this->quantity - $this->shipmentItems->sum('quantity'), 0);
    }

    /**
     * Check if item has purchasable
     * Twig is bad when it comes to separating functions from attributes and will always try to load
     * a function if attribute (even relationship) doesn't exist. This causes twig to load
     * the relationship and access through item.purchasable always will return false positive.
     *
     * @return bool
     */
    public function getHasPurchasableAttribute()
    {
        return !empty($this->purchasable);
    }

    /**
     * Check how much the value has been discounted on invoice
     *
     * @return float|int
     */
    public function getDiscountPercentageAttribute()
    {
        if ($this->unit_price->raw() == 0) {
            return 0;
        }

        return round(100 - (100 * $this->paid_price->raw() / $this->unit_price->raw()), 0);
    }

    /**
     * Access basket image
     * This has been added for order item as twig has issues with attempting to load a relationship
     *
     * @return string
     */
    public function getBasketImage()
    {
        if ($this->purchasable == null) {
            return '';
        }

        return $this->purchasable->getBasketImage();
    }

    /**
     * Check if item is shipped
     *
     * @return bool
     */
    public function isShipped() {
        return $this->quantity == $this->shipmentItems()->completed()->sum('quantity');
    }

    /**
     * Amend the order item price to a new value
     *
     * @param $new_value
     * @param $use_ex_vat
     */
    public function amendPrice($new_value, $use_ex_vat)
    {
        if ($use_ex_vat) {

            $this->fill([
                'unit_price_ex_vat' => $new_value,
                'paid_price_ex_vat' => $new_value,
            ]);

            if ($this->isDirty('paid_price_ex_vat')) {
                // Set value as null to avoid confusion due to price calculations as the alternate is not used
                $this->fill([
                    'unit_price' => null,
                    'paid_price' => null,
                ]);
            }
            return;
        }

        $this->fill([
            'unit_price' => $new_value,
            'paid_price' => $new_value,
        ]);

        if ($this->isDirty('paid_price')) {
            // Set value as null to avoid confusion due to price calculations as the alternate is not used
            $this->fill([
                'unit_price_ex_vat' => null,
                'paid_price_ex_vat' => null,
            ]);
        }
    }
}
