<?php

namespace Mtc\Orders\Factories;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Mtc\Orders\Item;
use Mtc\Orders\Contracts\OrderAmendInvoiceFactoryContract;
use Mtc\Checkout\Invoice;
use Mtc\Checkout\Invoice\Address;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Orders\Order;
use Mtc\Orders\OrderAmendState;
use Mtc\Orders\Surcharge;

/**
 * Class OrderAmendInvoiceFactory
 *
 * @package Mtc\Orders
 */
class OrderAmendInvoiceFactory implements OrderAmendInvoiceFactoryContract
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var Collection
     */
    protected $items;

    /**
     * @var array
     */
    protected $lines = [];

    /**
     * @var OrderAmendState
     */
    protected $state;

    /**
     * @var Request
     */
    protected $request;

    /**
     * OrderAmendInvoiceFactory constructor.
     *
     * @param OrderContract $order
     */
    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    /**
     * Prepare invoice changes
     */
    protected function prepareChanges()
    {
        return $this->prepareItems()
            || $this->prepareSurcharges();
    }

    /**
     * Create an invoice from order Amends
     */
    public function create(OrderContract $order, OrderAmendState $state)
    {
        $this->order = $order;
        $this->state = $state;
        if ($this->prepareChanges() === false) {
            return;
        }

        /** @var Invoice $invoice */
        $invoice = Invoice::query()
            ->create([
                'payable_type' => 'order',
                'payable_id' => $this->order->id,
                'due_at' => Carbon::now(),
                'amount' => $this->getAmount(),
                'amount_ex_vat' => $this->getAmount(false),
                'amount_in_currency' => $this->getAmount(),
                'use_ex_vat' => $this->order->use_ex_vat,
                'currency' => $this->order->currency->currency,
                'email' => $this->order->email,
                'member_id' => $this->order->member_id,
                'reference' => $this->order->id,
                'outstanding_amount' => $this->getAmount(),
                'outstanding_amount_ex_vat' => $this->getAmount(false),
                'vat_value' => $this->getVatAmount(),
                'template' => config('invoices.default_template'),
            ]);

        $this->items
            ->each(function ($item) use ($invoice) {
                $invoice->items()->create($item);
            });

        collect($this->lines)
            ->each(function ($line) use ($invoice) {
                $invoice->lines()->create($line);
            });

        $address = $this->order
            ->billingAddress
            ->only(array_keys(\Mtc\Foundation\Address::$blueprint));
        $address['type'] = Address::TYPE_RECIPIENT;
        $invoice->addresses()->create($address);
        $invoice->addresses()->create(config('invoices.seller_address'));
    }

    /**
     * Prepare items for processing
     *
     * @return bool
     */
    protected function prepareItems()
    {
        $this->items = collect($this->state->to['items'])
            ->filter(function ($item_values, $id) {
                return isset($item_values['quantity']) && $item_values['quantity'] > $this->state->from['items'][$id]['quantity'];
            })
            ->map(function ($item_values, $id) {
                $item = Item::query()->find($id);
                $quantity = $item_values['quantity'] - $this->state->from['items'][$id]['quantity'];

                return [
                    'purchasable_id' => $item->purchasable_id,
                    'purchasable_type' => $item->purchasable_type,
                    'parent_type' => Item::class,
                    'parent_id' => $id,
                    'name' => $item->name,
                    'original_value' => $item->unit_price->raw(true),
                    'original_value_ex_vat' => $item->unit_price->raw(false),
                    'value' => $item->paid_price->raw(true),
                    'value_ex_vat' => $item->paid_price->raw(false),
                    'quantity' => $quantity,
                    'line_total' => $quantity * $item->paid_price->raw(true),
                    'line_total_ex_vat' => $quantity * $item->paid_price->raw(false),
                    'vat_rate' => $item->vat_rate,
                    'details' => $item->attribute_fields,
                ];
            })
            ->filter();

        return $this->items->isNotEmpty();
    }

    /**
     * Prepare surcharges for invoice
     *
     * @return bool
     */
    protected function prepareSurcharges()
    {
        $changed_surcharges = collect($this->state->to['surcharges'])
            ->map(function ($surcharge, $index) {
                $original_data = $this->state->from['surcharges'][$index];
                return [
                    'is_negative' => 0,
                    'name' => $surcharge['name'] ?? $original_data['name'],
                    'type' => $surcharge['surcharge_type'] ?? $original_data['surcharge_type'],
                    'value' => $surcharge['surcharge_amount'] - $original_data['surcharge_amount'],
                    'parent_id' => $original_data['id'],
                    'parent_type' => Surcharge::class,
                ];
            })->filter(function ($line) {
                return $line['value'] > 0;
            });

        if ($changed_surcharges->isEmpty()) {
            return false;
        }

        $this->lines = array_merge($this->lines, $changed_surcharges->toArray());
        return true;
    }

    /**
     * Get the total amount for invoice
     *
     * @return mixed
     */
    protected function getAmount($vat = true)
    {
        $item_total = $vat ? $this->items->sum('line_total') : $this->items->sum('line_total_ex_vat');
        $surcharges = collect($this->lines)->sum('value');
        return $item_total + $surcharges;
    }

    /**
     * Get the vat for invoice
     *
     * @return mixed
     */
    protected function getVatAmount()
    {
        return $this->items
            ->sum(function ($item) {
                return $item['line_total'] * $item['vat_rate'];
            });
    }
}
