<?php

namespace Mtc\Orders;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Illuminate\Database\Eloquent\SoftDeletes;
use Mtc\Orders\Events\ShipmentCancelled;

/**
 * Class OrderShipment
 *
 * @package Mtc\Orders
 */
class OrderShipment extends Model
{
    use SoftDeletes;

    /**
     * Status constants
     * Used to define the state of the shipment.
     * Mainly used for split payments.
     */
    const STATUS_PENDING = 0;
    const STATUS_COMPLETED = 1;

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'order_id',
        'service_id',
        'status',
        'shipment_value',
        'exported_at',
        'courier_name',
        'tracking_number',
        'tracking_url',
        'label_format',
        'label',
        'shipment_data',
    ];

    /**
     * Dispatch events on model events
     *
     * @var array
     */
    protected $dispatchesEvents = [
        'deleted' => ShipmentCancelled::class,
    ];

    /**
     * @var array
     */
    protected $casts = [
        'exported_at' => 'datetime',
        'shipment_data' => 'array',
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        self::deleted(function ($self) {
            $self->boxes()->delete();
            $self->items()->delete();
        });
    }

    /**
     * Shipment items
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(OrderShipmentItem::class, 'shipment_id');
    }

    /**
     * Order for which this shipment is created
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Shipment boxes
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function boxes()
    {
        return $this->hasMany(OrderShipmentBox::class, 'shipment_id');
    }

    /**
     * Get the shipment reference
     *
     * @return mixed
     */
    public function getReferenceAttribute()
    {
        return App::makeWith('shipment_reference', [
            'shipment' => $this
        ]);
    }

    /**
     * Update the order of the shipment as shipped
     *
     * @param bool $only_if_part_shipped
     */
    public function setOrderAsShipped($only_if_part_shipped = false)
    {
        if ($only_if_part_shipped && $this->order->status_id != config('orders.statuses.part-shipped')) {
            return;
        }

        if ($this->order->isFullyShipped()) {
            $this->order->status_id = config('orders.statuses.shipped');
            $this->order->save();
        }
    }

    /**
     * Check if Shipment is valid and can be exported to courier
     *
     * @return bool
     */
    public function validates()
    {
        return true;
    }
}
