<?php

namespace Mtc\Orders\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Orders\Order;
use Mtc\Orders\Status;

/**
 * Class StatusController
 *
 * @package Mtc\Orders
 */
class StatusController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        /** @var Builder $query */
        $query = Status::query()->oldest('id');
        $this->page_meta['title'] = 'Manage Order Statuses';
        return template('/admin/order-statuses/index.twig', [
            'order_statuses' => $query->paginate(config('orders.admin.pagination')),
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Order Status';

        $twig_params = [
            'page_meta' => $this->page_meta,
            'order_status' => new Status(),
        ];

        return template('admin/order-statuses/edit.twig', $twig_params);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'name' => 'required'
        ]);

        $order_status = Status::query()->create($validated);
        $order_status->email_template_path = config('orders.status_default_email_template');
        $order_status->fill($request->input());
        $order_status->send_email_on_status = (!empty($request->input('send_email_on_status'))) ? 1 : 0;
        $order_status->is_failure_status = (!empty($request->input('is_failure_status'))) ? 1 : 0;
        $order_status->active = (!empty($request->input('active'))) ? 1 : 0;
        $order_status->save();

        session()->flash('success', 'Order Status created successfully');
        return redirect()->route('order-statuses.edit', $order_status->id);
    }

    /**
     * Show status email template.
     *
     * @param $id
     * @param OrderContract $order
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View|string
     */
    public function show($id, OrderContract $order)
    {
        $status = Status::findOrFail($id);
        if (!$status->email_template_path) {
            return 'Email template not specified';
        }
        //Get order information for template
        $order = $order->newQuery()
            ->paid()
            ->latest()
            ->first();

        //Return template
        return template($status->email_template_path, [
            'order' => $order,
            'order_status_text' => $status->getStatusText($order),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param $id
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit($id)
    {
        $this->page_meta['title'] = 'Edit Order Status';

        $twig_params = [
            'page_meta' => $this->page_meta,
            'order_status' => Status::find($id),
        ];

        return template('admin/order-statuses/edit.twig', $twig_params);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $id)
    {
        $validated = $this->validate($request, [
            'name' => 'required'
        ]);

        $order_status = Status::query()->find($id);
        $order_status->fill($request->input());
        $order_status->send_email_on_status = (!empty($request->input('send_email_on_status'))) ? 1 : 0;
        $order_status->is_failure_status = (!empty($request->input('is_failure_status'))) ? 1 : 0;
        $order_status->active = (!empty($request->input('active'))) ? 1 : 0;
        $order_status->save();

        session()->flash('success', 'Order Status updated');
        return redirect()->back();
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function list()
    {
        $this->page_meta['title'] = 'Order Emails';

        $order = Order::query()
            ->paid()
            ->latest()
            ->first();

        $emails = Status::query()
            ->where('send_email_on_status', 1)
            ->get()
            ->map(function ($status) use ($order) {
                return [
                    'title' => str_replace('{{SITE_NAME}}', config('app.name'), $status->email_subject),
                    'file_path' => $status->email_template_path,
                    'content' => template($status->email_template_path, [
                        'order' => $order,
                        'order_status_text' => $status->email_text,
                    ])
                ];
            });

        $twig_params = [
            'page_meta' => $this->page_meta,
            'header_style' => 'font-family: Helvetica,Arial,sans-serif;text-align:center;margin:40px 0 0;font-size:30px;',
            'emails' => $emails,
        ];

        return template('admin/order-statuses/email-list.twig', $twig_params);
    }
}
