<?php

namespace Mtc\Orders;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;
use Mtc\Money\Facades\Currency;

/**
 * Bank Transfer Payment Gateway
 *
 * @package  Mtc\BankTransfer
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class ManualOrderPayment implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        return App::make(config('orders.manual_order_applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $invoice = template('manual_orders/payment_form.twig', [
            'invoice' => $invoice
        ]);
        return new PaymentForm($invoice, 'template', [
            'name' => __('orders::orders.manual_order')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        $data = [
            'provider' => 'manual-order',
            'amount' => $invoice->getOutstandingAmount(),
            'reference' => config('orders.manual_order_reference_prefix') . $invoice->getId(),
            'details' => [],
            'confirmed_at' => now(),
        ];

        if (config('checkout.multi_currency_payments')) {
            $data['currency_code'] = $invoice->getCurrency();
            $data['amount_in_currency'] = Currency::inCurrency(
                $invoice->getOutstandingAmount(),
                $invoice->getCurrency()
            );
        }

        return $data;
    }
}
