<?php

namespace Mtc\Orders\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Orders\Contracts\ShipmentFactoryContract;
use Mtc\Orders\Http\Resources\OrderShipmentBoxResponse;
use Mtc\Orders\Order;
use Mtc\Orders\OrderShipment;
use Mtc\ShippingManager\Facades\Courier;

/**
 * Class OrderController
 *
 * @package Mtc\Orders
 */
class ShipmentController extends Controller
{
    /**
     *
     * @param $order_id
     * @param OrderContract $order
     */
    public function show(Request $request, $order_id, OrderShipment $shipment)
    {
        if ($request->wantsJson()) {
            return new OrderShipmentBoxResponse($shipment);
        }

        $order = $shipment->order;

        if (config('courier.enabled')) {
            $courier_services = Courier::getServices();
        }
        $this->page_meta['title'] = 'View Shipment For Order';
        return template('admin/orders/shipments/show.twig', [
            'order' => $order,
            'shipment' => $shipment,
            'page_meta' => $this->page_meta,
            'courier_services' => $courier_services ?? [],
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create($order_id, OrderContract $order)
    {
        $order = $order->find($order_id);
        $order->load([
            'shipments.items',
            'items.shipmentItems',
            'items.purchasable',
        ]);

        $this->page_meta['title'] = 'Create New Shipment For Order';
        return template('admin/orders/shipments/create.twig', [
            'order' => $order,
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, $order_id, OrderContract $order)
    {
        /** @var Order $order */
        $order = $order->find($order_id);

        App::make(ShipmentFactoryContract::class)->create($request, $order);

        session()->flash('success', 'Shipment Created');
        return redirect()->to(route('orders.show', [ $order_id ]));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($order_id, OrderShipment $shipment)
    {
        if (empty($shipment->exported_at)) {
            $shipment->delete();
            session()->flash('success', 'Shipment Cancelled');
        } else {
            session()->flash('success', 'Shipment Cannot be deleted as it is already Shipped');
        }
        return redirect(route('orders.show', [ $shipment->order_id ]));
    }

    /**
     * Update shipment as shipped
     *
     * @param Order $order
     * @param OrderShipment $shipment
     * @return \Illuminate\Http\RedirectResponse
     */
    public function complete(Order $order, OrderShipment $shipment)
    {
        $shipment->status = OrderShipment::STATUS_COMPLETED;
        $shipment->save();

        session()->flash('success', 'Shipment Updated');
        return redirect()->back();
    }

    /**
     * Picking list
     *
     * @param Order $order
     * @param OrderShipment $shipment
     * @return string
     */
    public function pickingList(Order $order, OrderShipment $shipment)
    {
        return template(config('orders.picking_list_template'), [
            'order' => $order,
            'shipment' => $shipment,
        ]);
    }
}
