<?php

namespace Mtc\Orders\Providers;

use Mtc\Orders\Events\OrderPaid;
use Mtc\Orders\Jobs\BulkOrderStatusChange;
use Illuminate\Foundation\Support\Providers\EventServiceProvider;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Event;
use Mtc\Checkout\Events\PaidInvoice;
use Mtc\Orders\Events\FetchOrderProcessingActions;
use Mtc\Orders\Events\OrderStatusChange;
use Mtc\Orders\Listeners\AttemptDeferredPaymentCapture;
use Mtc\Orders\Listeners\DeductStockOnPaidOrder;
use Mtc\Orders\Listeners\EditOrderAction;
use Mtc\Orders\Listeners\MarkOrderPaidFromInvoice;
use Mtc\Orders\Listeners\PrintInvoiceAction;
use Mtc\Orders\Listeners\ResendOrderConfirmationAction;
use Mtc\Orders\Listeners\SendOrderStatusEmail;
use Mtc\Orders\Listeners\UpdateOrderStatusOnRefund;
use Mtc\Orders\Listeners\ViewFailedPayments;
use Mtc\Orders\Listeners\ViewOrderSuccessPage;
use Mtc\Orders\Listeners\ViewUnpaidOrders;
use Mtc\Orders\OrderStatus;
use Mtc\Refunds\Events\RefundCreated;

/**
 * Class OrderEventServiceProvider
 *
 * @package Mtc\Money
 */
class OrderEventServiceProvider extends EventServiceProvider
{
    /**
     * The event handler mappings for the application.
     *
     * @var array
     */
    protected $listen = [
        FetchOrderProcessingActions::class => [
            ViewOrderSuccessPage::class,
            ResendOrderConfirmationAction::class,
            EditOrderAction::class,
            PrintInvoiceAction::class,
        ],
        OrderPaid::class => [
            DeductStockOnPaidOrder::class,
        ],
        'orders_index_extra_actions' => [
            ViewUnpaidOrders::class,
            ViewFailedPayments::class,
        ],
        RefundCreated::class => [
            UpdateOrderStatusOnRefund::class
        ]
    ];

    /**
     * Boot the provider
     */
    public function boot()
    {
        parent::boot();

        Event::listen(
            OrderStatusChange::class,
            config('orders.send_order_status_email_listener', SendOrderStatusEmail::class)
        );
        Event::listen(
            OrderPaid::class,
            config('orders.send_order_status_email_listener', SendOrderStatusEmail::class)
        );
        Event::listen(PaidInvoice::class, config('orders.mark_paid_listener', MarkOrderPaidFromInvoice::class));

        if (config('orders.auto_capture_deferred_payments_on_status') > 0) {
            Event::listen(OrderStatusChange::class, AttemptDeferredPaymentCapture::class);
        }

        if ($this->app->runningInConsole() === false) {
            $statuses = Cache::remember('active-order-statuses', now()->addDay(), function () {
                return OrderStatus::query()
                    ->active()
                    ->get();
            });

            $statuses->each(function ($status) {
                Event::listen('order_bulk_actions', function () use ($status) {
                    return [
                        'name' => __('orders::orders.admin.bulk_change_order_status', ['status' => $status->name]),
                        'value' => 'status-' . $status->id,
                        'handler' => BulkOrderStatusChange::class,
                    ];
                });
            });
        }
    }
}
