<?php

namespace Mtc\Orders\Factories;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Orders\Contracts\ShipmentFactoryContract;
use Mtc\Orders\Order;
use Mtc\Orders\OrderShipment;

/**
 * Class ShipmentFactory
 *
 * Create a shipment on order based on request values
 *
 * @package Mtc\Orders
 */
class ShipmentFactory implements ShipmentFactoryContract
{
    /**
     * Create a shipment
     *
     * @param Request $request
     * @param Order $order
     * @return mixed
     */
    public function create(Request $request, OrderContract $order)
    {
        $shipment_value = $this->calculateValue($request, $order);

        $shipment = $order->shipments()
            ->create([
                'shipment_value' => $shipment_value,
                'order_value' => $shipment_value,
                'status' => OrderShipment::STATUS_COMPLETED,
            ]);

        collect($request->input('quantity'))
            ->each(function ($quantity, $id) use ($shipment, $request) {
                $shipment->items()
                    ->create([
                        'order_item_id' => $id,
                        'quantity' => $quantity ?? 0,
                        'ship_weight' => $request->input("weight.{$id}", 0),
                    ]);
            });

        $order->refresh();
        if ($order->isFullyShipped()) {
            $order->changeStatus(config('orders.statuses.shipped'), Auth::user());
        } else {
            $order->changeStatus(config('orders.statuses.part-shipped'), Auth::user());
        }
    }

    /**
     * Calculate shipment value
     *
     * @param Request $request
     * @param Order $order
     * @return array|mixed|string|null
     */
    protected function calculateValue($request, $order)
    {
        if (config('order.manual_shipment_value') && $request->input('order_value') > 0) {
            return $request->input('order_value');
        }

            return collect($order->items)
                ->map(function ($item) use ($request) {
                    return $request->input("quantity.{$item->id}", 0) * $item->paid_price->raw();
                })
                ->sum();
    }
}
