<?php

namespace Mtc\Orders\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Invoice\Payment as PaymentModel;
use Mtc\Checkout\Facades\Payment;

/**
 * Class CaptureDeferredPayment
 *
 * @package Mtc\Orders\Jobs
 */
class CaptureDeferredPayment implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * @var PaymentModel
     */
    protected $payment;

    /**
     * @var InvoiceRepositoryContract
     */
    protected $invoice_repository;

    /**
     * Create a new job instance.
     *
     * @param array $orders
     * @param string $action
     * @return void
     */
    public function __construct(PaymentModel $payment)
    {
        $this->payment = $payment;
        $this->invoice_repository = App::make(InvoiceRepositoryContract::class);
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        try {
            Payment::setActiveDriver($this->payment->provider);
            Payment::chargeDeferredPayment($this->payment);

            $this->payment->confirmed_at = now();
            $this->payment->save();

            $this->invoice_repository->load($this->payment->invoice_id);
            $this->invoice_repository->reduceOutstandingAmount($this->payment->amount);
            $this->invoice_repository->checkIfInvoiceIsPaid();
        } catch (\Exception $exception) {
            $this->payment->failed_at = now();
            $this->payment->failure_details = $exception->getMessage();
            $this->payment->save();
            Log::warning('Failed deferred charge', [
                'payment' => $this->payment,
                'exception' => $exception,
            ]);
        }
    }
}
