<?php

namespace Mtc\Orders;

use Illuminate\Support\Collection;
use Illuminate\Support\Str;

/**
 * Class OrderAmendState
 *
 * State class that stores information about changes on order during edit
 *
 * @package Mtc\Orders
 */
class OrderAmendState
{
    /**
     * @var array
     */
    public $from = [
        'details' => [],
        'items' => [],
        'discounts' => [],
        'surcharges' => [],
        'addresses' => [],
    ];

    /**
     * @var array
     */
    public $to = [
        'details' => [],
        'items' => [],
        'discounts' => [],
        'surcharges' => [],
        'addresses' => [],
    ];


    /**
     * Get the instance as an array.
     *
     * @return Collection
     */
    public function changeList()
    {
        if (!empty($this->to['details']['cost_total'])) {
            $this->to['details']['cost_total'] = round($this->to['details']['cost_total'], 2);
        }

        if (!empty($this->to['details']['cost_subtotal'])) {
            $this->to['details']['cost_subtotal'] = round($this->to['details']['cost_subtotal'], 2);
        }

        $changes = collect($this->to['details']);

        collect($this->to['items'])
            ->filter()
            ->each(function ($item_amends, $id) use ($changes) {
                if ($item_amends['quantity'] == 0) {
                    $changes->push('Removed ' . $this->from['items'][$id]['name']);
                    return;
                }

                if (empty($this->from['items'][$id])) {
                    $amends = collect($item_amends)
                        ->only([
                            'paid_price',
                            'paid_price_ex_vat',
                            'quantity',
                            'percentage_discount',
                            'amount_discount'
                        ])
                        ->filter();
                    $changes->put('New item ' . $item_amends['name'], json_encode($amends));
                    return;
                }

                $amends = collect($item_amends)->filter()->toArray();
                $changes->put($this->from['items'][$id]['name'], json_encode($amends));
            });

        collect($this->to['addresses'])
            ->filter()
            ->each(function ($address, $type) use ($changes) {
                $changes->put($type . ' address', implode(', ', $address));
            });

        collect($this->to['discounts'])
            ->filter()
            ->each(function ($discount, $id) use ($changes) {
                $changes->put($this->from['discounts'][$id]['name'], json_encode($discount));
            });

        collect($this->to['surcharges'])
            ->filter()
            ->each(function ($surcharge, $id) use ($changes) {
                $changes->put($this->from['surcharges'][$id]['name'], json_encode($surcharge));
            });

        return $changes
            ->filter()
            ->map(function ($changed_values, $changed_field) {
                if (stripos($changed_values, 'removed') !== false) {
                    return Str::title(str_replace(['-', '_'], ' ', $changed_values));
                }
                return Str::title(str_replace(['-', '_'], ' ', $changed_field . ' set to ' . $changed_values));
            });
    }

    /**
     * Check if there are state changes
     *
     * @return bool
     */
    public function hasChanges(): bool
    {
        return collect($this->to)
            ->reject(function ($values, $section) {
                if (empty($values)) {
                    return true;
                }

                if ($section === 'details') {
                    return false;
                }

                return collect($values)->filter()->isNotEmpty();
            })
            ->isNotEmpty();
    }
}
