<?php

namespace Mtc\Orders;

use Illuminate\Support\Collection;
use Mtc\Members\Member;
use Mtc\Orders\Contracts\OrderRepositoryContract;

/**
 * Class OrderRepository
 *
 * @package Mtc\Orders
 */
class OrderRepository implements OrderRepositoryContract
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * Fetch Attribute from payable
     *
     * @param $attribute
     * @return mixed
     */
    public function getAttribute($attribute)
    {
        return $this->order->getAttribute($attribute);
    }

    /**
     * Attach model to the repository
     *
     * @param $model
     * @return void
     */
    public function setModel($model)
    {
        $this->order = $model;
    }


    /**
     * Get the ID of payable to ensure we can link to it
     *
     * @return int
     */
    public function getPayableClass(): string
    {
        return 'order';
    }

    /**
     * Get the reference string
     *
     * @return string
     */
    public function getReference(): string
    {
        return config('checkout.transaction_prefix', '') . $this->order->id;
    }

    /**
     * Get the invoice template used by the site for this payable
     *
     * @return string
     */
    public function getTemplate(): string
    {
        return config('invoices.default_template');
    }

    /**
     * Get the payment amount
     *
     * @return float
     */
    public function getAmount(): float
    {
        return $this->order->cost_total->raw(true);
    }

    /**
     * Get the invoice items
     *
     * @return array
     */
    public function getItems(): Collection
    {
        return $this->order->items;
    }

    /**
     * Get the additional lines for the invoice
     *
     * @return array
     */
    public function getLines(): array
    {
        return array_merge(
            $this->order->surcharges
                ->map(function ($surcharge) {
                    return [
                        'is_negative' => 0,
                        'name' => $surcharge->name,
                        'type' => $surcharge->surcharge_type,
                        'value' => $surcharge->surcharge_amount->raw(),
                        'value_ex_vat' => $surcharge->surcharge_amount->withoutTax(),
                        'parent_id' => $surcharge->id,
                        'parent_type' => \Mtc\Orders\Surcharge::class,
                    ];
                })->toArray(),
            $this->order->discounts
                ->map(function ($discount) {
                    return [
                        'is_negative' => 1,
                        'name' => $discount->name,
                        'type' => $discount->discount_type,
                        'value' => $discount->discount_amount->raw(),
                        'value_ex_vat' => $discount->discount_amount->withoutTax(),
                        'parent_id' => $discount->id,
                        'parent_type' => \Mtc\Orders\Discount::class,
                    ];
                })->toArray()
        );
    }

    /**
     * Get the billing address
     *
     * @return array
     */
    public function getAddress(): array
    {
        return $this->order
            ->billingAddress
            ->only(array_keys(\Mtc\Foundation\Address::$blueprint));
    }

    /**
     * Get the additional info that invoice should know about this payable
     *
     * @return mixed
     */
    public function getAdditionalDetails()
    {
        return null;
    }

    /**
     * Get the member assigned to this payable
     *
     * @return Member
     */
    public function getMember()
    {
        if (class_exists(\Mtc\Members\Providers\MemberProvider::class)) {
            return $this->order->member;
        }
        return null;
    }

    /**
     * Check whether invoice should use inc vat or ex vat values
     *
     * @return bool
     */
    public function useExVat(): bool
    {
        return !empty($this->order->use_ex_vat);
    }
}
