<?php

namespace Mtc\Paypal;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\ExternalPaymentGateway;
use Mtc\Checkout\Contracts\HandlesRefunds;
use Mtc\Checkout\Contracts\HasDeferredCharges;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Checkout\PaymentForm;

/**
 * PayPal Payment Gateway
 *
 * @package  Mtc\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class PayPal implements ExternalPaymentGateway, HasDeferredCharges, HandlesRefunds
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param BasketRepositoryInterface $basket
     * @return bool
     */
    public function isApplicable(BasketRepositoryInterface $basket): bool
    {
        return App::make(config('paypal.applicable_check_class'))->handle($basket);
    }

    /**
     * Render the payment template.
     *
     * @param BasketRepositoryInterface $basket
     * @return string
     */
    public function getPaymentForm(BasketRepositoryInterface $basket1): PaymentForm
    {
        return new PaymentForm('');
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        /** @var PayPalService $service */
        $service = App::make(PayPalService::class);
        return $service->charge($request, $invoice);
    }

    /**
     * Check if payment can have a deferred charge
     *
     * @param Payment $payment
     * @return bool
     */
    public function allowDeferredCharge(Payment $payment)
    {
        if (config('checkout.deferred_payments') !== true) {
            return false;
        }

        /** @var PayPalService $service */
        $service = App::make(PayPalService::class);
        return $service->isChargeable($payment);
    }

    /**
     * Charge a payment that was set up as deferred
     *
     * @param Payment $payment
     * @return bool
     */
    public function chargeDeferredPayment(Payment $payment)
    {
        if (config('checkout.deferred_payments') !== true) {
            return false;
        }

        /** @var PayPalService $service */
        $service = App::make(PayPalService::class);
        return $service->deferredCharge($payment);
    }

    /**
     * Check if payment is refundable by payment gateway
     *
     * @param Payment $payment
     * @return bool|array
     */
    public function isRefundable(Payment $payment)
    {
        /** @var PayPalService $service */
        $service = App::make(PayPalService::class);
        return $service->isRefundable($payment);
    }

    /**
     * Process a refund on payment
     *
     * @param Payment $payment
     * @param null|double $amount
     * @return bool|array
     */
    public function refund(Payment $payment, $amount = null)
    {
        /** @var PayPalService $service */
        $service = App::make(PayPalService::class);
        return $service->refund($payment, $amount);
    }
}
