<?php

namespace Mtc\Paypal\Providers;

use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use Mtc\Basket\Http\Resources\BasketAjax;
use Mtc\Paypal\Console\Commands\PayPalInstall;
use Mtc\Paypal\Contracts\PayPalBasketFillInterface;
use Mtc\Paypal\Contracts\PayPalPaymentFactoryInterface;
use Mtc\Paypal\Listeners\AddPayPalButtonToBasket;
use Mtc\Paypal\PayPal;
use PayPal\Rest\ApiContext;

/**
 * Class PaypalServiceProvider
 *
 * @package Mtc\Stripe\Providers
 */
class PaypalServiceProvider extends ServiceProvider
{
    /**
     * Bootstrap the application services.
     */
    public function boot()
    {
        if (config('paypal.enabled') === false) {
            return;
        }

        app('twig_loader')->addPath(dirname(__DIR__, 2) . '/resources/templates');

        $this->app['payment']->extend('paypal', function($app) {
            return new PayPal();
        });

        if ($this->app->runningInConsole()) {

            $this->commands([
                PayPalInstall::class
            ]);

            $this->publishAssets();
        } else {
            Event::listen(BasketAjax::class, AddPayPalButtonToBasket::class);
        }
    }

    /**
     * Register the application services.
     */
    public function register()
    {
        // Automatically apply the package configuration
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/paypal.php', 'paypal');

        if (config('paypal.enabled') === false) {
            return;
        }

        Route::middleware('web')->group(dirname(__DIR__, 2) . '/routes/web.php');

        // Load a custom basket (or default).
        $this->app->singleton(PayPalPaymentFactoryInterface::class, function ($app) {
            return $this->app->make(config('paypal.payment_factory'));
        });

        // Load a custom basket (or default).
        $this->app->singleton(PayPalBasketFillInterface::class, function ($app) {
            return $this->app->make(config('paypal.basket_fill_factory'));
        });

        $this->app->bind(ApiContext::class, function () {
            $context = new \PayPal\Rest\ApiContext(
                new \PayPal\Auth\OAuthTokenCredential(
                    config('paypal.client_id'),
                    config('paypal.client_secret')
                )
            );

            if (config('paypal.live')) {
                $context->setConfig([
                    'mode' => 'live'
                ]);
            }

            return $context;
        });
    }

    /**
     * Publish assets
     */
    public function publishAssets()
    {
        $this->publishes([
            dirname(__DIR__, 2) . '/config/paypal.php' => config_path('paypal.php'),
        ], 'config');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/templates' => resource_path('templates'),
        ], 'templates');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/assets' => public_path('assets'),
        ], 'assets');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/less' => resource_path('less/vendor'),
        ], 'assets');
    }

}
