<?php

namespace Mtc\Paypal\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\InvoiceFactoryContract;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableFactoryContract;
use Mtc\Checkout\Facades\Payment;
use Mtc\Paypal\Contracts\PayPalPaymentFactoryInterface;
use Mtc\Paypal\PayPalService;
use PayPal\Rest\ApiContext;

/**
 * Class PayPalController
 *
 * @package Mtc\Paypal
 */
class PayPalController extends Controller
{
    /**
     * Create a payment and redirect to paypal for authorization
     *
     * @param BasketRepositoryInterface $basket
     */
    public function index(BasketRepositoryInterface $basket, ApiContext $context)
    {
        /** @var PayPalPaymentFactoryInterface $payment_charger */
        $payment_charger = App::make(PayPalPaymentFactoryInterface::class);
        try {
            /** @var \PayPal\Api\Payment $payment */
            $payment = $payment_charger->create($basket);
            $payment->create($context);
            $approval_url = $payment->getApprovalLink();
            App::make(PayPalPaymentFactoryInterface::class)
                ->createPaymentReference($payment->getToken(), $basket->getCostTotalAttribute());

            return redirect()->to($approval_url);

        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), [ $exception ]);
            return redirect()->to(route('failed_payment'));
        }
    }

    /**
     * Payment authorization page
     *
     * @param Request $request
     * @param PayPalService $rest
     * @param ApiContext $context
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function return(Request $request, PayPalService $pay_pal_service, ApiContext $context)
    {
        try {
            $basket = $pay_pal_service->prepareAuthorize($request);
            return App::make(config('paypal.controller_response_class'))->authorize($request, $basket);

        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), [ $exception ]);
            return redirect()->to(route('failed_payment'));
        }
    }

    /**
     * Charge the payment
     *
     * @param Request $request
     * @param PayableFactoryContract $payable_factory
     * @param InvoiceFactoryContract $invoice_factory
     * @throws \Exception
     */
    public function charge(Request $request, PayableFactoryContract $payable_factory, InvoiceFactoryContract $invoice_factory)
    {
        if (!App::make(BasketRepositoryInterface::class)->getModel()->exists()) {
            throw new \Exception('Basket does not exist');
        }
        $payable = $payable_factory->create($request);
        $invoice_repository = $invoice_factory->create($payable);

        Payment::setActiveDriver('paypal');

        // Only try charging if order is not paid
        if ($invoice_repository->getOutstandingAmount() == 0) {
            return $this->checkAndRedirectPaidInvoice($request, $invoice_repository);
        }

        try {
            $payment_details = Payment::charge($request, $invoice_repository);
            $invoice_repository->savePayment($payment_details);
            return redirect()->to(URL::signedRoute('successful_payment', ['id' => $invoice_repository->getId() ], null, config('app.env') === 'production'));
        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), [ $exception ]);
            return redirect()->to(route('failed_payment'));
        }

    }

    /**
     * User cancelled payment
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function cancel() {
        return redirect()->to(route('failed_payment'));
    }

    /**
     * Deal with an already paid invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return \Illuminate\Http\RedirectResponse
     */
    protected function checkAndRedirectPaidInvoice(Request $request, InvoiceRepositoryContract $invoice)
    {
        // Allow viewing success page if basket matches session
        if (App::make(config('checkout.verify_user_viewing_paid_order'))->verify($request, $invoice)) {
            return redirect()->to(URL::signedRoute('successful_payment', ['id' => $invoice->getId() ], null, config('app.env') === 'production'));
        }

        return redirect()->to('/');
    }
}
