<?php

namespace Mtc\PayPalPayments\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mtc\PayPalPayments\PayPalSettings;

class PayPalApi
{
    /**
     * @var Client
     */
    protected $api;

    /**
     * @var string
     */
    private $client_id;

    /**
     * @var string
     */
    private $client_secret;

    public function __construct(Client $api, string $client_id, string $client_secret)
    {
        $this->api = $api;
        $this->client_id = $client_id;
        $this->client_secret = $client_secret;
    }

    /**
     * GET Request
     *
     * @param $path
     * @param int $depth
     * @return mixed
     * @throws \Exception
     */
    public function get($path, $auth_assertion = true, $depth = 1)
    {
        try {
            $headers = [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->getAccessToken(),
            ];
            if ($auth_assertion) {
                $headers['PayPal-Auth-Assertion'] = $this->authAssertionHeader();
            }

            $result = $this->api->get(
                $this->endpoint($path),
                [
                    'headers' => $headers,
                ]
            );
        } catch (ConnectException $exception) {
            if ($depth < 3) {
                return $this->get($path, $auth_assertion, $depth + 1);
            }
            throw new \Exception('Unable to connect to PayPal API');
        }
        return json_decode((string)$result->getBody());
    }

    /**
     * POST request
     *
     * @param $path
     * @param array $data
     * @param array $extra_headers
     * @param int $depth
     * @return mixed
     * @throws \Exception
     */
    public function post($path, array $data = [], $extra_headers = [], $depth = 0)
    {
        $context = [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->getAccessToken(),
                'Paypal-Partner-Attribution-Id' => $this->attributionId(),
            ],
        ];

        if ($this->shouldIncludeAuthAssertionHeader($path)) {
            $context['headers']['PayPal-Auth-Assertion'] = $this->authAssertionHeader();
        }

        if (!empty($extra_headers)) {
            $context['headers'] = array_merge($context['headers'], $extra_headers);
        }

        if (!empty($data)) {
            $context['body'] = json_encode($data);
        }

        Log::info('Request to' . $path, $context);

        try {
            $result = $this->api->post($this->endpoint($path), $context);
        } catch (ConnectException $exception) {
            if ($depth < 3) {
                return $this->post($path, $data, $extra_headers, $depth + 1);
            }
            throw new \Exception('Unable to connect to PayPal API');
        }

        Log::debug(
            'PayPal API POST Request',
            [
                'endpoint' => $path,
                'data' => $data,
                'response' => json_decode((string)$result->getBody(), true),
            ]
        );
        return json_decode((string)$result->getBody(), true);
    }

    /**
     * @param $path
     * @return string
     */
    protected function endpoint($path): string
    {
        if (in_array(Config::get('app.env'), ['demo', 'production']) == false) {
            return 'https://api-m.sandbox.paypal.com/' . ltrim($path, '/');
        }

        if (Config::get('paypal_payments.use_sandbox')) {
            return 'https://api-m.sandbox.paypal.com/' . ltrim($path, '/');
        }
        return 'https://api-m.paypal.com/ ' . ltrim($path, '/');
    }

    /**
     * @return string
     */
    private function getAccessToken(): string
    {
        return Cache::remember(
            'paypal-api-access-token',
            now()->addHours(7),
            function () {
                return $this->oauthAccessToken();
            }
        );
    }

    /**
     * @return string
     */
    private function oauthAccessToken(): string
    {
        $result = $this->api->post(
            $this->endpoint('v1/oauth2/token'),
            [
                'auth' => [
                    $this->client_id,
                    $this->client_secret,
                ],
                'form_params' => [
                    'grant_type' => 'client_credentials'
                ],
            ]
        );

        return json_decode((string)$result->getBody())->access_token;
    }

    /**
     * @return string
     */
    private function attributionId(): string
    {
        return Config::get('paypal_payments.partner_attribution_id');
    }

    /**
     * Check if PayPal-Auth-Assertion header needs to be included
     * This is breaking the partner referral registration link
     *
     * @param string $path
     * @return bool
     */
    protected function shouldIncludeAuthAssertionHeader(string $path): bool
    {
        return Str::contains($path, [ 'partner-referrals' ]) === false;
    }

    /**
     * Build auth assertion header to request
     *
     * @return string
     */
    protected function authAssertionHeader(): string
    {
        $settings = App::make(PayPalSettings::class);
        $merchant_id = $settings->get('merchant_id');
        $client_id = $settings->get('API_CLIENT_ID');
        $auth_string = '{"iss":"' . $client_id . '","payer_id":"' . $merchant_id . '"}';
        return base64_encode('{"alg":"none"}') . '.' . base64_encode($auth_string) . '.';
    }
}
