<?php

namespace Mtc\PayPalPayments\Listeners;

use Illuminate\Support\Facades\Config;
use Mtc\PayPalPayments\PayPalSettings;
use Mtc\PayPalPayments\Services\PaymentService;

class AddPayPalScripts
{
    /**
     * @var PayPalSettings
     */
    private $settings;

    /**
     * @var PaymentService
     */
    private $service;

    public function __construct(PayPalSettings $settings, PaymentService $service)
    {
        $this->settings = $settings;
        $this->service = $service;
    }

    /**
     * Set the delivery name for order when creating it
     *
     * @param $event
     * @return string
     */
    public function handle($event)
    {
        $merchant_id = $this->settings->get('merchant_id');
        $client_id = $this->settings->get('API_CLIENT_ID');

        if (empty($client_id)) {
            return '';
        }

        $currency = 'GBP';
        $partner_attribution_id = Config::get('paypal_payments.partner_attribution_id');
        $client_token = $this->service->generateClientToken();
        $enable_funding = $this->getEnableFunding();
        $components = implode(',', $this->getComponents(!empty($enable_funding)));
        return "<script src='https://www.paypal.com/sdk/js?components=$components&client-id={$client_id}"
            . "&merchant-id={$merchant_id}&currency={$currency}&intent=capture$enable_funding'
                data-client-token='{$client_token}'
                data-partner-attribution-id='{$partner_attribution_id}'
                data-merchant-id='{$merchant_id}'></script>";
    }

    protected function getComponents(bool $requires_funding): array
    {
        $components = [
            'hosted-fields',
            'buttons',
        ];

        if ($requires_funding) {
            $components[] = 'marks';
            $components[] = 'funding-eligibility';
        }

        if ($this->settings->get('pay_later')) {
            $components[] = 'messages';
        }

        return $components;
    }

    protected function getEnableFunding()
    {
        $funding = '';
        $funding_methods = [];
        if ($this->settings->get('bancontact')) {
            $funding_methods[] = 'bancontact';
        }
        if ($this->settings->get('blik')) {
            $funding_methods[] = 'blik';
        }
        if ($this->settings->get('eps')) {
            $funding_methods[] = 'eps';
        }
        if ($this->settings->get('giropay')) {
            $funding_methods[] = 'giropay';
        }
        if ($this->settings->get('ideal')) {
            $funding_methods[] = 'ideal';
        }
        if ($this->settings->get('mybank')) {
            $funding_methods[] = 'mybank';
        }
        if ($this->settings->get('przelewy24')) {
            $funding_methods[] = 'p24';
        }
        if ($this->settings->get('sofort')) {
            $funding_methods[] = 'sofort';
        }

        if (!empty($funding_methods)) {
            $funding = '&enable-funding=' . implode(',', $funding_methods);
        }

        return $funding;
    }
}
