<?php

namespace Mtc\PayPalPayments\Services;

use Carbon\Carbon;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\PayPalPayments\Exceptions\InstrumentDeclinedException;
use Mtc\PayPalPayments\Maps\MapInvoiceRepositoryToPayPal;
use Mtc\PayPalPayments\PayPalSettings;

class PaymentService
{
    /**
     * @var PayPalApi
     */
    private $api;

    /**
     * @var MapInvoiceRepositoryToPayPal
     */
    private $invoice_map;

    public function __construct(PayPalApi $api, MapInvoiceRepositoryToPayPal $invoice_map)
    {
        $this->api = $api;
        $this->invoice_map = $invoice_map;
    }

    /**
     * @param InvoiceRepositoryContract $invoice
     * @return mixed
     * @throws \Exception
     */
    public function createOrder(InvoiceRepositoryContract $invoice)
    {
        return $this->api->post('v2/checkout/orders', $this->invoice_map->map($invoice));
    }

    /**
     * @param $order_id
     * @return mixed
     * @throws \Exception
     */
    public function getOrder($order_id)
    {
        return $this->api->get('v2/checkout/orders/' . $order_id);
    }

    /**
     * @param string $order_id
     * @return array
     * @throws InstrumentDeclinedException
     */
    public function capture(string $order_id)
    {
        $headers = [
            'PayPal-Request-Id' => 'CAPTURE-' . $order_id . microtime(true),
        ];
        $response = $this->api->post('v2/checkout/orders/' . $order_id . '/capture', [], $headers);
        if ($response['status'] === 'COMPLETED') {
            return [
                'id' => $response['id'],
                'status' => $response['status'],
                'capture_id' => $response['purchase_units'][0]['payments']['captures'][0]['id'],
                'capture_status' => $response['purchase_units'][0]['payments']['captures'][0]['status'],
                'amount' => $response['purchase_units'][0]['payments']['captures'][0]['amount']['value'],
                'currency_code' => $response['purchase_units'][0]['payments']['captures'][0]['amount']['currency_code'],
            ];
        }
        if (($response['details'][0]['issue'] ?? '') === InstrumentDeclinedException::CODE) {
            throw new InstrumentDeclinedException();
        }
        throw new \Exception($response['details'][0]['description'] ?? 'Payment Capture Failed');
    }

    /**
     * @param string $capture_id
     * @param double $amount
     * @param string $currency
     * @param string|int $reference
     * @return mixed
     * @throws \Exception
     */
    public function refund(string $capture_id, $amount, $currency, $reference)
    {
        $payload = [
            'amount' => [
                'currency_code' => $currency,
                'value' => $amount,
            ],
            'invoice_number' => $reference
        ];

        $endpoint = 'v2/payments/captures/' . $capture_id . '/refund';
        return $this->api->post($endpoint, $payload, $this->refundHeaders());
    }

    /**
     * @return string
     */
    public function generateClientToken()
    {
        return Cache::remember('paypal-client-token', Carbon::now()->addMinutes(30), function () {
            $response = $this->api->post('/v1/identity/generate-token');
            return $response['client_token'];
        });
    }

    /**
     * Check order response 3D secure details
     *
     * @param array $order_details
     * @return bool
     */
    public function verify3DSecure($order_details): bool
    {
        $settings = App::make(PayPalSettings::class);
        if ($settings->get('3DSEC_ENABLED') === false) {
            return true;
        }

        return $settings->get($this->inputTo3DSecScenario($order_details)) === 'accept';
    }

    /**
     * Convert PayPal response data to 3D secure scenario
     *
     * @param $order_details
     * @return string
     */
    protected function inputTo3DSecScenario($order_details): string
    {
        Log::debug('PayPal 3D Secure details on payment', [
            $order_details
        ]);

        if (!isset($order_details->payment_source->card->authentication_result)) {
            Log::warning('Order without payment authorization data', [
                $order_details
            ]);
            return '3DSEC_unknown';
        }

        $auth_result = $order_details->payment_source->card->authentication_result;
        $enrollment = strtolower($auth_result->three_d_secure->enrollment_status ?? '-');
        $authentication =  strtolower($auth_result->three_d_secure->authentication_status ?? '-');
        $liability = strtolower(substr($auth_result->liability_shift, 0, 1));
        return '3DSEC_' . $enrollment . $authentication . $liability;
    }

    /**
     * @return string[]
     */
    protected function refundHeaders(): array
    {
        $idempotency_key = config('app.site_id') . microtime(true);

        return [
            'PayPal-Request-Id' => $idempotency_key,
        ];
    }
}
