<?php

namespace Mtc\PayPalPayments\Tests\Unit;

use Carbon\Carbon;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Mtc\PayPalPayments\Models\PaypalSettingModel;
use Mtc\PayPalPayments\PayPalSettings;
use Mtc\PayPalPayments\Tests\TestCase;

class PayPalSettingTest extends TestCase
{
    use DatabaseMigrations;

    public function testHasAccess()
    {
        self::assertFalse($this->settings()->hasAccess());

        $this->settings()->save('merchant_id', 'foo-baz-bar');
        self::assertTrue($this->settings()->hasAccess());
    }

    public function testReadyToTakePayment()
    {
        self::assertFalse($this->settings()->readyToTakeCardPayment());

        $this->settings()->save('merchant_id', 'foo-baz-bar');
        $this->settings()->save('payments_receivable', 1);
        $this->settings()->save('primary_email_confirmed', 1);
        $this->settings()->save('CUSTOM_CARD_PROCESSING', 'ACTIVE');

        self::assertTrue($this->settings()->readyToTakeCardPayment());

        $this->settings()->clearSettings('merchant_id');
        self::assertFalse($this->settings()->readyToTakeCardPayment());

        $this->settings()->save('merchant_id', 'foo-baz-bar');
        $this->settings()->clearSettings('payments_receivable');
        self::assertFalse($this->settings()->readyToTakeCardPayment());

        $this->settings()->save('payments_receivable', 1);
        $this->settings()->clearSettings('primary_email_confirmed');
        self::assertFalse($this->settings()->readyToTakeCardPayment());

        $this->settings()->save('primary_email_confirmed', 1);
        $this->settings()->clearSettings('payments_receivable');
        self::assertFalse($this->settings()->readyToTakeCardPayment());

        $this->settings()->save('payments_receivable', 1);
        $this->settings()->clearSettings('CUSTOM_CARD_PROCESSING');
        self::assertFalse($this->settings()->readyToTakeCardPayment());

        $this->settings()->save('CUSTOM_CARD_PROCESSING', 'ACTIVE');
        $this->settings()->save('CUSTOM_CARD_LIMIT_TYPE', 'anything');
        self::assertFalse($this->settings()->readyToTakeCardPayment());
    }

    public function testCardPaymentsEnabled()
    {
        self::assertTrue($this->settings()->cardPaymentsEnabled());

        $this->settings()->save('PPCP_ENABLED', 1);
        self::assertTrue($this->settings()->cardPaymentsEnabled());

        $this->settings()->toggle('PPCP_ENABLED');
        self::assertFalse($this->settings()->cardPaymentsEnabled());
    }

    public function testSignupDefaults()
    {
        $defaults = $this->settings()->getSignupDefaults();

        self::assertArrayHasKey('website', $defaults);
        self::assertArrayHasKey('phone', $defaults);
        self::assertArrayHasKey('phone_country_code', $defaults);
    }

    public function testSavingSetting()
    {
        $this->settings()->save('TEST_KEY', 'test-value');

        $test_array = [
            'foo',
            'baz',
            'bar',
        ];

        $this->settings()->save('TEST_KEY', $test_array);

        $exists = PaypalSettingModel::query()
            ->where('key', 'TEST_KEY')
            ->where('value', 'test-value')
            ->exists();

        self::assertTrue($exists);

        $exists_array = PaypalSettingModel::query()
            ->where('key', 'TEST_KEY')
            ->where('data', '["foo","baz","bar"]')
            ->exists();

        self::assertTrue($exists_array);
    }

    public function testGettingAndUpdatingValue()
    {
        self::assertEquals(null, $this->settings()->get('TEST_KEY'));

        $this->settings()->save('TEST_KEY', 'test-value');
        self::assertEquals('test-value', $this->settings()->get('TEST_KEY'));

        $this->settings()->save('TEST_KEY', 'foo baz bar');
        self::assertEquals('foo baz bar', $this->settings()->get('TEST_KEY'));
    }

    public function testClearAllSettings()
    {
        $this->settings()->save('TEST_KEY', 'test-value');
        $this->settings()->save('foo', 'test-value');
        $this->settings()->save('baz', 'test-value');
        $this->settings()->save('bar', 'test-value');

        $has_settings = PaypalSettingModel::query()->exists();
        self::assertTrue($has_settings);

        $this->settings()->clear();

        $has_settings = PaypalSettingModel::query()->exists();
        self::assertFalse($has_settings);

    }

    public function testClearPatternSettings()
    {
        $this->settings()->save('TEST_foo', 'test-value');
        $this->settings()->save('TEST_baz', 'test-value');
        $this->settings()->save('TEST_bar', 'test-value');
        $this->settings()->save('foo', 'test-value');
        $this->settings()->save('baz', 'test-value');
        $this->settings()->save('bar', 'test-value');

        $setting_count = PaypalSettingModel::query()->count();
        self::assertEquals(6, $setting_count);

        $this->settings()->clearSettings('TEST_');

        $setting_count = PaypalSettingModel::query()->count();
        self::assertEquals(3, $setting_count);

        self::assertNull($this->settings()->get('TEST_foo'));
        self::assertNull($this->settings()->get('TEST_baz'));
        self::assertNull($this->settings()->get('TEST_bar'));

    }

    public function testTogglingSettings()
    {
        $this->settings()->save('TEST_KEY', 1);
        self::assertEquals(1, $this->settings()->get('TEST_KEY'));

        $this->settings()->toggle('TEST_KEY');
        self::assertEquals(0, $this->settings()->get('TEST_KEY'));

        $this->settings()->toggle('TEST_KEY');
        self::assertEquals(1, $this->settings()->get('TEST_KEY'));
    }

    public function testVettingChecks()
    {
        $this->settings()->save('PPCP_CUSTOM_VETTING', 'NEED_MORE_DATA');
        self::assertTrue($this->settings()->needsVetting());
        self::assertFalse($this->settings()->vettingReview());
        self::assertFalse($this->settings()->vettingDenied());

        $this->settings()->save('PPCP_CUSTOM_VETTING', 'IN_REVIEW');
        self::assertFalse($this->settings()->needsVetting());
        self::assertTrue($this->settings()->vettingReview());
        self::assertFalse($this->settings()->vettingDenied());


        $this->settings()->save('PPCP_CUSTOM_VETTING', 'DENIED');
        self::assertFalse($this->settings()->needsVetting());
        self::assertFalse($this->settings()->vettingReview());
        self::assertTrue($this->settings()->vettingDenied());

        $expected = Carbon::now()->addDays(90)->format('jS F Y');
        self::assertEquals($expected, $this->settings()->vettingRetryDate());
    }

    public function testLimitWarnings()
    {
        self::assertFalse($this->settings()->limitWarning());
        self::assertFalse($this->settings()->overLimitWarning());

        $this->settings()->save('PPCP_CUSTOM_VETTING', 'SUBSCRIBED');
        $this->settings()->save('CUSTOM_CARD_PROCESSING', 'ACTIVE');
        $this->settings()->save('CUSTOM_CARD_LIMIT_TYPE', 'GENERAL');

        self::assertTrue($this->settings()->limitWarning());
        self::assertFalse($this->settings()->overLimitWarning());

        $this->settings()->save('WITHDRAW_MONEY_LIMIT', 500);
        $this->settings()->save('SEND_MONEY_LIMIT', 500);

        self::assertFalse($this->settings()->limitWarning());
        self::assertTrue($this->settings()->overLimitWarning());
    }

    protected function settings(): PayPalSettings
    {
        return $this->app->make(PayPalSettings::class);
    }
}