<?php

namespace Mtc\PayPalPayments;

use Illuminate\Database\Eloquent\HigherOrderBuilderProxy;

class PayPalSettings
{
    /**
     * Default values for settings
     * Used unless different value is saved in DB
     *
     * @var array
     */
    protected static $default_setting_values = [
        '3DSEC_ENABLED' => 1,
        'PPCP_ENABLED' => true,
        '3DSEC_yyp' => 'accept',
        '3DSEC_ynn' => 'decline',
        '3DSEC_yrn' => 'decline',
        '3DSEC_yap' => 'accept',
        '3DSEC_yuu' => 'decline',
        '3DSEC_yun' => 'decline',
        '3DSEC_ycu' => 'decline',
        '3DSEC_y-n' => 'decline',
        '3DSEC_n-n' => 'accept',
        '3DSEC_u-n' => 'accept',
        '3DSEC_u-u' => 'decline',
        '3DSEC_b-n' => 'accept',
        '3DSEC_--u' => 'decline',
        'button_layout' => 'vertical',
        'button_size' => 'default',
    ];

    /**
     * @var PayPalSettingDBRepository
     */
    protected $repository;

    /**
     * @param PayPalSettingDBRepository $repository
     */
    public function __construct(PayPalSettingDBRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * @return bool
     */
    public function hasAccess(): bool
    {
        return !empty($this->repository->get('merchant_id'));
    }

    /**
     * @return bool
     */
    public function allowPayments()
    {
        return $this->hasAccess()
            && $this->repository->get('payments_receivable')
            && $this->repository->get('primary_email_confirmed');
    }

    /**
     * @return bool
     */
    public function readyToTakeCardPayment()
    {
        return $this->hasAccess()
            && $this->repository->get('payments_receivable')
            && $this->repository->get('primary_email_confirmed')
            && $this->repository->get('CUSTOM_CARD_PROCESSING') === 'ACTIVE'
            && empty($this->repository->get('CUSTOM_CARD_LIMIT_TYPE'));
    }

    /**
     * @return bool
     */
    public function cardPaymentsEnabled(): bool
    {
        if ($this->repository->get('PPCP_ENABLED') === null) {
            return self::$default_setting_values['PPCP_ENABLED'];
        }

        return $this->repository->get('PPCP_ENABLED') == 1;
    }

    /**
     * @return array
     */
    public function getSignupDefaults()
    {
        return [
            'website' => config('app.url'),
            'phone' => config('settings.TELEPHONE_NUMBER'),
            'phone_country_code' => 44
        ];
    }

    /**
     * @param $key
     * @return bool|HigherOrderBuilderProxy|mixed|null
     */
    public function get($key)
    {
        $value = $this->repository->get($key);
        if ($value !== null) {
            return $value;
        }

        if ($key === 'PPCP_ENABLED') {
            return $this->readyToTakeCardPayment();
        }

        return self::$default_setting_values[$key] ?? null;
    }

    /**
     * @param $key
     * @param $value
     */
    public function save($key, $value): void
    {
        $this->repository->save($key, $value);
    }

    /**
     * @param $pattern
     */
    public function clearSettings($pattern): void
    {
        $this->repository->clearStartingWith($pattern);
    }

    /**
     * @param $key
     */
    public function toggle($key): void
    {
        $new_value = $this->repository->get($key) == 1 ? 0 : 1;
        $this->save($key, $new_value);
    }

    public function clear(): void
    {
        $this->repository->clearAll();
    }

    /**
     * @return bool
     */
    public function needsVetting(): bool
    {
        return $this->repository->get('PPCP_CUSTOM_VETTING') === 'NEED_MORE_DATA';
    }

    /**
     * @return bool
     */
    public function vettingReview(): bool
    {
        return $this->repository->get('PPCP_CUSTOM_VETTING') === 'IN_REVIEW';
    }

    /**
     * @return bool
     */
    public function vettingDenied(): bool
    {
        return $this->repository->get('PPCP_CUSTOM_VETTING') === 'DENIED';
    }

    /**
     * PayPal doesn't provide info whether this denied vetting gives you
     * reapply date, so we have to improvise and return
     * date as 90 days after vetting was denied
     *
     * @return string
     */
    public function vettingRetryDate(): string
    {
        return $this->repository->updateTime('PPCP_CUSTOM_VETTING')
            ->addDays(90)
            ->format('jS F Y');
    }

    /**
     * @return bool
     */
    public function limitWarning(): bool
    {
        return $this->repository->get('PPCP_CUSTOM_VETTING') === 'SUBSCRIBED'
            && $this->repository->get('CUSTOM_CARD_PROCESSING') === 'ACTIVE'
            && $this->repository->get('CUSTOM_CARD_LIMIT_TYPE') === 'GENERAL'
            && empty($this->repository->get('WITHDRAW_MONEY_LIMIT'))
            && empty($this->repository->get('SEND_MONEY_LIMIT'));
    }

    /**
     * @return bool
     */
    public function overLimitWarning(): bool
    {
        return $this->repository->get('PPCP_CUSTOM_VETTING') === 'SUBSCRIBED'
            && $this->repository->get('CUSTOM_CARD_PROCESSING') === 'ACTIVE'
            && $this->repository->get('CUSTOM_CARD_LIMIT_TYPE') === 'GENERAL'
            && !empty($this->repository->get('WITHDRAW_MONEY_LIMIT'))
            && !empty($this->repository->get('SEND_MONEY_LIMIT'));
    }

    /**
     * @param $merchant_status
     */
    public function saveMerchantStatus($merchant_status): void
    {
        $vetting_status = collect($merchant_status->products)
                ->firstWhere('name', 'PPCP_CUSTOM')
                ->vetting_status ?? null;

        $this->save('PPCP_CUSTOM_VETTING', $vetting_status);

        if (!empty($merchant_status->capabilities)) {
            $custom_card_processing = collect($merchant_status->capabilities)
                    ->firstWhere('name', 'CUSTOM_CARD_PROCESSING')
                    ->status ?? null;

            $custom_card_limit_type = collect($merchant_status->capabilities)
                    ->firstWhere('name', 'CUSTOM_CARD_PROCESSING')
                    ->limits[0]
                    ->type ?? null;
            $withdraw_money_limit = collect($merchant_status->capabilities)
                    ->firstWhere('name', 'WITHDRAW_MONEY')
                    ->limits ?? null;

            $send_money_limit = collect($merchant_status->capabilities)
                    ->firstWhere('name', 'SEND_MONEY')
                    ->limits ?? null;

            $this->save('CUSTOM_CARD_PROCESSING', $custom_card_processing);
            $this->save('CUSTOM_CARD_LIMIT_TYPE', $custom_card_limit_type);
            $this->save('WITHDRAW_MONEY_LIMIT', $withdraw_money_limit);
            $this->save('SEND_MONEY_LIMIT', $send_money_limit);
        }

        $this->save('payments_receivable', $merchant_status->payments_receivable);
        $this->save('primary_email', $merchant_status->primary_email);
        $this->save('primary_email_confirmed', $merchant_status->primary_email_confirmed);
        $this->save('scopes', $merchant_status->oauth_integrations[0]->oauth_third_party[0]->scopes);
    }
}
