<?php

namespace Mtc\PayPalPayments\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mtc\PayPalPayments\PayPalSettings;

class ReferralSignupService
{
    /**
     * @var PayPalApi
     */
    private $api;

    /**
     * @var PayPalMicroServiceApi
     */
    private $microservice;

    /**
     * @var PayPalSettings
     */
    private $settings;

    public function __construct(PayPalApi $api, PayPalMicroServiceApi $microservice, PayPalSettings $settings)
    {
        $this->api = $api;
        $this->microservice = $microservice;
        $this->settings = $settings;
    }

    /**
     * @param Request $request
     * @return mixed|string
     * @throws \Exception
     */
    public function createReferral(Request $request)
    {
        $response = $this->api->post('v2/customer/partner-referrals', $this->mapData($request));

        return collect($response['links'])
                ->where('rel', 'action_url')
                ->first()['href'] ?? '';
    }

    /**
     * @param string $merchant_id
     * @return mixed
     * @throws \Exception
     */
    public function getMerchantStatus(string $merchant_id)
    {
        $partner_id = config('paypal_payments.partner_id');
        return $this->api->get("v1/customer/partners/{$partner_id}/merchant-integrations/{$merchant_id}", false);
    }

    /**
     * @param $tracking_id
     * @return mixed
     * @throws \Exception
     */
    public function getSignupStatus($tracking_id)
    {
        $partner_id = config('paypal_payments.partner_id');
        return $this->api->get(
            "/v1/customer/partners/{$partner_id}/merchant-integrations?tracking_id={$tracking_id}",
            false
        );
    }

    /**
     * @param string $merchant_id
     */
    public function attachMerchantId(string $merchant_id)
    {
        $this->settings->save('merchant_id', $merchant_id);
        try {
            $this->microservice->post('/set-merchant', ['merchant_id' => $merchant_id]);
        } catch (\Exception $exception) {
            Log::error('Failed to notify paypal.mtcassets.com merchant_id', [$exception->getMessage()]);
        }
    }

    /**
     * @param Request $request
     * @return array
     */
    private function mapData(Request $request): array
    {
        return [
            "business_entity" => [
                "names" => [
                    [
                        "business_name" => $request->input('business_name'),
                        "type" => "LEGAL_NAME"
                    ]
                ],
                "emails" => [
                    [
                        "type" => "CUSTOMER_SERVICE",
                        "email" => $request->input('email')
                    ]
                ],
                "website" => $request->input('website'),
                "phones" => [
                    [
                        "country_code" => $request->input('phone_country_code'),
                        "national_number" => $request->input('phone'),
                        "type" => "CUSTOMER_SERVICE"
                    ]
                ],
            ],
            "email" => $request->input('email'),
            "preferred_language_code" => "en-US",
            "tracking_id" => $this->trackingId(),
            "partner_config_override" => [
                "partner_logo_url" => "https://projects.mtcmedia.co.uk/images/logo.jpg",
                "return_url" => route('admin.paypal.referral-onboarded'),
                "return_url_description" => "the url to return the merchant after the paypal onboarding process.",
                "show_add_credit_card" => true
            ],
            "legal_consents" => [
                [
                    "type" => "SHARE_DATA_CONSENT",
                    "granted" => (bool)$request->input('share_data_consent')
                ]
            ],
            "operations" => [
                [
                    "operation" => "API_INTEGRATION",
                    "api_integration_preference" => [
                        "rest_api_integration" => [
                            "integration_method" => "PAYPAL",
                            "integration_type" => "THIRD_PARTY",
                            "third_party_details" => [
                                "features" => [
                                    "PAYMENT",
                                    "REFUND",
                                    "ACCESS_MERCHANT_INFORMATION",
                                ]
                            ]
                        ],
                    ],
                ],
            ],
            "products" => [
                $request->input('ppcp_consent') !== 'false' ? "PPCP" : "EXPRESS_CHECKOUT",
            ]
        ];
    }

    /**
     * @return string
     */
    private function trackingId()
    {
        $site_name = config('app.name');
        $site_domain = config('app.url');

        try {
            $account = $this->microservice->account($site_name, $site_domain);
            $this->settings->save('PP_SERVICE_API_KEY', $account->api_token);
            $this->settings->save('SIGNUP_TRACKING_ID', $account->tracking_id);
            return $account->tracking_id;
        } catch (\Exception $exception) {
            Log::error('Failed to get tracking ID from paypal service, using fallback', [$exception->getMessage()]);
            $tracking_id = $this->fallbackTrackingId($site_name);
            $this->settings->save('SIGNUP_TRACKING_ID', $account->tracking_id);
            return $tracking_id;
        }
    }

    /**
     * Structure should be the same, but we are not able to store this on the service
     * This means we are not able to process webhooks
     *
     * @param string $site_name
     * @return string
     */
    private function fallbackTrackingId(string $site_name): string
    {
        return 'mtc-' . Str::slug($site_name) . '-' . Str::random(8);
    }
}
