<?php

namespace Mtc\PayPalPayments\Maps;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\PayPalPayments\PayPalSettings;

class MapInvoiceRepositoryToPayPal
{
    /**
     * @var InvoiceRepositoryContract
     */
    protected $invoice;

    /**
     * @var PayPalSettings
     */
    protected $settings;

    /**
     * @var string
     */
    protected $delivery;

    public function __construct(PayPalSettings $settings)
    {
        $this->settings = $settings;
    }

    /**
     * Map Data
     *
     * @param InvoiceRepositoryContract $invoice
     * @return array
     */
    public function map(InvoiceRepositoryContract $invoice): array
    {
        $this->invoice = $invoice;
        $this->delivery = $this->checkDelivery();
        $data =  [
            'intent' => 'CAPTURE',
            'application_context' => [
                'shipping_preference' => $this->delivery !== 'DELIVERY' ? 'NO_SHIPPING' :  'SET_PROVIDED_ADDRESS',
                'locale' => 'en-GB',
                'brand_name' => config('app.name'),
                'user_action' => 'PAY_NOW',
                'return_url' => route('successful_payment'),
                'cancel_url' => route('checkout.index'),
            ],
            'purchase_units' => [
                0 => [
                    'reference_id' => '1',
                    'amount' => $this->amountInfo(),
                    'payee' => [
                        'merchant_id' => $this->settings->get('merchant_id'),
                    ],
                    'items' => $this->items(),
                    'payment_instruction' => [
                        'disbursement_mode' => 'INSTANT',
                    ],
                    'invoice_id' => $this->invoiceId(),
                    'custom_id' => $this->invoice->getId(),
                    'description' => 'Order with ' . config('app.name'),
                    'soft_descriptor' => config('app.name'),
                ],
            ],
            'payer' => $this->payerDetails(),
        ];

        if ($this->delivery !== 'NO_SHIPPING') {
            $data['purchase_units'][0]['shipping'] = $this->shippingDetails();
        }

        return $data;
    }

    /**
     * @return string
     */
    protected function invoiceId(): string
    {
        return Str::slug(config('app.site_id')) . '-' . $this->invoice->getReference() . '-' . Carbon::now('U');
    }

    /**
     * @return array
     */
    protected function shippingDetails(): array
    {
        if ($this->delivery === 'NO_SHIPPING') {
            return [];
        }

        $address = $this->invoice->getPayable()->shippingAddress;
        if ($address === null) {
            $address = $this->invoice->getPayable()->collectionAddress;

            $collection_surcharge = $this->invoice->getPayable()->surcharges()
                ->where('surcharge_type', 'collection')
                ->first();
            $delivery_name = 'S2S' . ($collection_surcharge->name ?? '');
            return [
                'name' => [
                    'full_name' => $delivery_name,
                ],
                'address' => [
                    'address_line_1' => $address->address1,
                    'address_line_2' => $address->address2 ?? '',
                    'postal_code' => $address->postcode ?? '',
                    'admin_area_2' => $address->city ?? '',
                    'country_code' => $address->country,
                    'admin_area_1' => $address->county ?? '',
                ],
            ];
        }
        $delivery_name = $this->invoice->getPayable()->deliverySurcharge->name;

        return [
            'method' => $delivery_name,
            'name' => [
                'full_name' => $address->first_name . ' ' . $address->last_name,
            ],
            'address' => [
                'address_line_1' => $address->address1,
                'address_line_2' => $address->address2 ?? '',
                'postal_code' => $address->postcode ?? '',
                'admin_area_2' => $address->city ?? '',
                'country_code' => $address->country,
                'admin_area_1' => $address->county ?? '',
            ],
        ];
    }

    /**
     * @return Collection
     */
    protected function items(): Collection
    {
        return collect($this->invoice->getModel()->items)
            ->map(function ($invoice_item) {
                return [
                    'name' => $invoice_item->name,
                    'sku' => $invoice_item->purchasable->getSku(),
                    'unit_amount' => [
                        'currency_code' => 'GBP',
                        'value' => $invoice_item->value->raw(false, 2),
                    ],
                    'tax' => [
                        'currency_code' => 'GBP',
                        'value' => round($invoice_item->value->tax(), 2),
                    ],
                    'quantity' => $invoice_item->quantity,
                ];
            });
    }

    /**
     * @return array
     */
    protected function amountInfo(): array
    {
        $item_sum = $this->items()
            ->sum(function ($item) {
                return $item['tax']['value'] * $item['quantity'];
            });

        $data = [
            'currency_code' => 'GBP',
            'value' => $this->invoice->getOutstandingAmount(),
            'breakdown' => [
                'item_total' => [
                    'currency_code' => 'GBP',
                    'value' => round($this->items()
                        ->sum(function ($item) {
                            return $item['unit_amount']['value'] * $item['quantity'];
                        }), 2),
                ],
                'tax_total' => [
                    'currency_code' => 'GBP',
                    'value' => round($item_sum, 2),
                ],
            ],
        ];

        if ($this->delivery === 'DELIVERY') {
            $data['breakdown']['shipping'] = [
                'currency_code' => 'GBP',
                'value' => round($this->invoice->getPayable()->deliverySurcharge->surcharge_amount->raw(true), 2) ?? 0,
            ];
        }
        return $data;
    }

    /**
     * @return string
     */
    protected function checkDelivery(): string
    {
        if ($this->invoice->getPayable()->deliverySurcharge) {
            return 'DELIVERY';
        }

        $collection_surcharge = $this->invoice->getPayable()->surcharges()
            ->where('surcharge_type', 'collection')
            ->first();

        return $collection_surcharge !== null
            ? 'COLLECTION'
            : 'NO_SHIPPING';
    }

    /**
     * @return array[]
     */
    protected function payerDetails(): array
    {
        $address = $this->invoice->getPayable()->billingAddress;
        return [
            'address' => [
                'address_line_1' => $address->address1,
                'address_line_2' => $address->address2 ?? '',
                'admin_area_2' => $address->city ?? '',
                'admin_area_1' => $address->county ?? '',
                'postal_code' => $address->postcode ?? '',
                'country_code' => $address->country,
            ],
            'name' => [
                'given_name' => $address->first_name,
                'surname' => $address->last_name,
            ]
        ];
    }
}
