<?php
/**
 * @author: Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @version: 10.09.2024
 */

namespace App\Models;

use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Mtc\Modules\Members\Models\Member;
use Mtc\Plugins\Clinic\Src\Logger;
use Mtc\Shop\Order;
use stdClass;

/**
 * Accessors
 *
 * @property int $id
 * @property int $member_id
 * @property string $loggable
 * @property int $loggable_id
 * @property string $subject
 * @property string $value
 * @property string $actor
 * @property int $actor_id
 * @property string $ip_address
 * @property stdClass $details
 *
 * @property Carbon $created_at
 * @property Carbon $updated_at
 */
class EventLog extends Model
{

    protected $table = 'event_log';

    protected $fillable = [
        'member_id',
        'loggable',
        'loggable_id',
        'subject',
        'value',
        'actor',
        'actor_id',
        'details',
        'ip_address',
    ];

    protected $casts = [
        'details' => 'object'
    ];

    /**
     * @return MorphTo
     */
    public function loggable() : MorphTo
    {
        return $this->morphTo(__FUNCTION__, 'loggable', 'loggable_id');
    }

    /**
     * @return MorphTo
     */
    public function actor() : MorphTo
    {
        return $this->morphTo(__FUNCTION__, 'actor', 'actor_id');
    }

    /**
     * @return BelongsTo
     */
    public function member() : BelongsTo
    {
        return $this->belongsTo(Member::class, 'member_id');
    }

    /**
     * Normalises event values for easy output
     *
     * @return void
     */
    public function normalise(): void
    {
        try {
            if ($this->subject === Logger::ORDER_STATUS) {
                $this->action = Logger::ACTION_UPDATED;
            } elseif ($this->subject === Logger::ORDER_ITEM_APPROVAL) {
                $this->action = Logger::ACTION_UPDATED;
            } elseif ($this->subject === Logger::REFUND) {
                $this->action = Logger::ACTION_CREATED;
            } else {
                $this->action = $this->value;
            }

            if ($this->subject === Logger::ORDER_STATUS) {
                $this->description = Order::$statuses[$this->value];
            } elseif ($this->subject === Logger::ORDER_ITEM_APPROVAL) {
                $this->action = Order\Item::$approved_statuses[$this->value];
                $this->description = $this->details->item_name;
                if (!empty($this->details->variation)) {
                    $this->description .= '<br />' . $this->details->variation;
                }
            } elseif ($this->subject === Logger::TREATMENT) {
                $this->description = $this->details->item_name;
                if (!empty($this->details->variation)) {
                    $this->description .= '<br />' . $this->details->variation;
                }
                $this->description .= '<br />';
                $this->description .= Carbon::parse($this->details->date_from)->format('d/m/Y');
                $this->description .= ' - ';
                $this->description .= Carbon::parse($this->details->date_to)->format('d/m/Y');
            } elseif ($this->subject === Logger::FILE) {
                $downloadLink = route('admin-patient-event-log-get-file', [
                    'id' => $this->id,
                ]);
                $this->description = $this->details->title;
                $this->description .= '<br />';
                $this->description .= '<a href="' . $downloadLink . '" target="_blank">File</a>';
            }
        } catch (Exception) {

        }
    }
}
