<?php

namespace Mtc\Plugins\NHS\Commands;

use Illuminate\Console\Command;
use Mtc\Modules\Members\Classes\MemberManager;
use Mtc\Modules\Members\Models\Member;
use Mtc\Plugins\NHS\Classes\Prescription;
use Mtc\Shop\Order as ShopOrder;


class NhsToolboxCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'nhs:toolbox';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = '...';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    private const NHS_TOOLBOX_ACTION_CREATE_PRESCRIPTION = 'create-prescription';
    private const NHS_TOOLBOX_ACTION_CREATE_ORDER = 'create-order';
    private const NHS_TOOLBOX_ACTION_CREATE_PATIENT = 'create-patient';

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $actions = [
            self::NHS_TOOLBOX_ACTION_CREATE_PRESCRIPTION,
            self::NHS_TOOLBOX_ACTION_CREATE_ORDER,
            self::NHS_TOOLBOX_ACTION_CREATE_PATIENT,
        ];

        $action = $this->choice('What would you like to do?', $actions, 0);

        switch ($action) {
            case self::NHS_TOOLBOX_ACTION_CREATE_PRESCRIPTION:
                $this->info("Creating prescription...");
                $prescription = $this->createPrescription();
                $this->info("Created prescription id:{$prescription->id}.");
                break;

            case self::NHS_TOOLBOX_ACTION_CREATE_ORDER:
                $this->info("Creating order...");
                $order = $this->createOrder();
                $this->info("Created order id:{$order->id}.");
                break;

            case self::NHS_TOOLBOX_ACTION_CREATE_PATIENT:
                $this->info("Creating patient...");
                $member = $this->createPatient();
                $this->info("Created member id:{$member->id}.");
                break;

            default:
                $this->info("Unknown action [{$action}]. Doing nothing...");
                break;
        }


        $this->info('Bye.');
    }


    private function createPrescription()
    {
        $prescription = null;

        $member_id = (int)$this->ask('Members ID?');

        $member = Member::find($member_id);

        if (! $member) return $prescription;

        $number_of_items = (int)$this->ask('How many items?', 1);

        $prescription_items = [];
        for ($number_of_items; $number_of_items--; $number_of_items > 0) {
            $temp_item = [];
            $temp_item['name'] = $this->ask('Item name:', 'Item ' . uniqid());
            $temp_item['quantity'] = $this->ask('Item qty:', 1);
            $prescription_items[] = $temp_item;
        }

        $prescription = self::createPrescriptionFromInput($member, $prescription_items);

        return $prescription;
    }


    private function createOrder()
    {
        $shop_order = null;

        $prescription_id = $this->ask('Prescription ID?');
        $this->info("Creating order...");

        $prescription = Prescription::find($prescription_id);

        if (! $prescription) return $shop_order;

        $shop_order = self::createOrderFromPrescription($prescription);

        return $shop_order;
    }


    private function createPatient()
    {
        $member = null;

        $data = [];
        $data['email'] = $this->ask('Email:', 'john.smith.' . uniqid() . '@test.com');
        $data['contact_no'] = '';
        $data['nhs_number'] = $this->ask('NHS No.:', 'NHS-' . uniqid());
        $data['billing_firstname'] = $this->ask('Forename:', 'John');
        $data['billing_lastname'] = $this->ask('Surname:', 'Smith');
        $data['billing_address1'] = $this->ask('A1:', '123 Street');
        $data['billing_address2'] = $this->ask('A2:', '');
        $data['billing_city'] = $this->ask('City:', 'City');
        $data['billing_postcode'] = $this->ask('Postcode:', 'AB1 2CD');
        $data['billing_country'] = 'GB';
        $data['no_password'] = true;

        $allow_no_email = false;

        $member_manager = new MemberManager();
        $member = $member_manager->save($data, new Member(), $allow_no_email);

        if (! $member) {
            print_r($member_manager->getErrors());
        }

        return $member;
    }


    public static function createOrderFromPrescription(Prescription $prescription)
    {
        $basket = $prescription->create_basket();

        ShopOrder::newOrder($basket);
        $shop_order = ShopOrder::find($basket->order_id);
        $shop_order->legacy()->markPaid();

        return $shop_order;
    }


    public static function createPrescriptionFromInput(Member $member, array $item_data)
    {
        $prescription = Prescription::create($member);
        $processed_items = $prescription->save_items_from_input($item_data);

        return $prescription;
    }
}
