<?php
/**
 * File Details
 * @package    mtc ecommerce
 * @subpackage splibvalidation
 * @author     Darren Marshall <darren@silentprogress.co.uk>
 * @copyright  Copyright (c) 2007, Darren Marshall
 * @version    1.0 (03/24/2007)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU General Public License
 *
 */

/**
 * Class Details
 * @package    mtc ecommerce
 * @subpackage splibvalidation
 * @author     Darren Marshall <darren@silentprogress.co.uk>
 * @copyright  Copyright (c) 2007, Darren Marshall
 * @version    1.0 (03/24/2007)
 * @license    http://www.gnu.org/copyleft/gpl.html GNU General Public License
 *
 */
class general_validation
{
    /**
     * Validates a string as an email address
     * Uses regular expressions to check for 'user@hostname.domain'
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_email($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/"                       // start pattern
                     . "^[a-z0-9_-]+"           // valid chars (at least once)
                     . "(\.[a-z0-9_-]+)*"       // dot valid chars (0-n times)
                     . "@"                      // at
                     . "[a-z0-9][a-z0-9-]*"     // valid chars (at least once)
                     . "(\.[a-z0-9-]+)*"        // dot valid chars (0-n times)
                     . "\.([a-z]{2,6})$"        // dot valid chars
                     . "/i";                    // end pattern, case insensitive

            return $this->do_test($reg_ex, $string);
        }
    }

    /**
     * Validates general form input from a textfield.
     * Uses regular expressions to allow the letters A-Z,
     * the numbers 0-9, a hyphen (-), a dot (.), ana a space ( )
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_general_text($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/^[0-9a-zA-Z\.\-\ ]+$/";

            return $this->do_test($reg_ex, $string);
        }

    }

     /**
     * Validates a UK postcode using regular expressions.
     * Validates formats: AN NAA, ANN NAA, AAN NAA, AANN NAA
     * ANA NAA, AANA NAA: (A=A to Z, N = 0 to 9) and
     * GIR 0AA, a unique Postcode that was issued.
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_UK_postcode($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $string = strtoupper(str_replace(' ','',$string));
            if($this->do_test("/^[A-Z]{1,2}[0-9]{2,3}[A-Z]{2}$/", $string) ||
               $this->do_test("/^[A-Z]{1,2}[0-9]{1}[A-Z]{1}[0-9]{1}[A-Z]{2}$/", $string) ||
               $this->do_test("/^GIR0[A-Z]{2}$/", $string))
            { return true; } else { return false; }
        }
    }

    /**
     * Validates a filename.
     * Uses regular expressions to allow the start letter A-Z,
     * and contain a-Z, 0-9, underscore(_), dash(-), and dot(.)
     * strings beginning with anything but a-Z or 0-9 will return false
     * and strings containing anything other than above mentioned will also
     * return false
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_filename($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/^[a-zA-Z0-9]+([a-zA-Z\_0-9\.-]*)$/";

            return $this->do_test($reg_ex, $string);
        }

    }

    /**
     * Validates a string as being numeric only.
     * Uses regular expressions to allow the numbers 0-9
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_numeric($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/^[0-9]+$/";

            return $this->do_test($reg_ex, $string);
        }

    }

    /**
     * Validates a string as being numeric only.
     * Uses regular expressions to allow the numbers 0-9 and spaces
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_numeric_allow_spaces($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/^[0-9\ ]+$/";

            return $this->do_test($reg_ex, $string);
        }

    }

    /**
     * Validates a string as being a decimal number only.
     * Uses regular expressions to allow the numbers 0-9 and (.) dot
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_decimal_number($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/^[0-9\.]+$/";

            return $this->do_test($reg_ex, $string);
        }

    }

    /**
     * Validates a string as being text only.
     * Uses regular expressions to allow the letters A-Z case insensitive
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_textonly($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/^[A-Za-z]+$/";

            return $this->do_test($reg_ex, $string);
        }

    }

    /**
     * Validates a string as being text only.
     * Uses regular expressions to allow the letters A-Z case insensitive
     * and also allow spaces
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_textonly_allow_spaces($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/^[A-Za-z\ ]+$/";

            return $this->do_test($reg_ex, $string);
        }

    }

    /**
     * Validates a string as being a name.
     * Uses regular expressions to allow the letters A-Z case insensitive
     * and also allow spaces and single quotes
     *
     * @param string $string the string to parse
     *
     * @return boolean the status of the validation
     */
    public function validate_name($string)
    {
        if ($this->is_empty($string)) { return false; } else {
            $reg_ex = "/^[A-Za-z-_\'\ ]+$/";

            return $this->do_test($reg_ex, $string);
        }

    }

    /**
     * Simple helper method to check if a string is empty.
     *
     * @param string $string the string to check
     *
      * @return boolean the status of the validation
     */
    public function is_empty($string)
    {
        if (trim($string)=="") { return true; } else { return false; }
    }

    /**
     * Private internal function to check the regular expression against the input string.
     *
     * @param string $reg_ex the regular expression to use
     * @param string $string the string to check
     *
      * @return boolean the status of the check
     */
    public function do_test($reg_ex, $string)
    {
        if (!preg_match($reg_ex, $string)) {
            return false;
        } else {
            return true;
        }
    }

}
