<?php

namespace Tests\Feature;

use Mtc\Shop\Item as ShopItem;
use App\ProductGroup;
use App\ShopHelper;


class ProductGroupsTest extends \PHPUnit\Framework\TestCase
{

    private static $product_names = [
        'atov'        => 'Atovaquone and Proguanil 250mg/100mg Single Tablet (Malarone generic tablets)',
        'malarone'    => 'Malarone Tablets 250/100mg Anti Malaria - Single Tablet',
        'maloff'      => 'Maloff Protect Tablets',

        'atov_p'      => 'Atovaquone and Proguanil Paediatric 62.5mg/25mg Single Tablet (equivalent To Malarone Paediatric)',
        'malarone_p'  => 'Malarone Paediatric Tablets 62.5/25mg Anti Malaria - Single Tablet',

        'doxy'        => 'Doxycycline 100mg (Single Capsules)',
        'lariam'      => 'Lariam (Mefloquine 250mg) Single Tablets',

        'another'      => 'Another Product',
    ];


    public static function dataTestProductGroupNameDetection()
    {
        return [
            [ 'assertSame', ProductGroup::GROUP_A, self::$product_names['atov'] ],
            [ 'assertSame', ProductGroup::GROUP_A, self::$product_names['malarone'] ],
            [ 'assertSame', ProductGroup::GROUP_A, self::$product_names['maloff'] ],

            [ 'assertSame', ProductGroup::GROUP_AP, self::$product_names['atov_p'] ],
            [ 'assertSame', ProductGroup::GROUP_AP, self::$product_names['malarone_p'] ],

            [ 'assertSame', ProductGroup::GROUP_D, self::$product_names['doxy'] ],

            [ 'assertSame', ProductGroup::GROUP_L, self::$product_names['lariam'] ],

            [ 'assertNotSame', ProductGroup::GROUP_A, self::$product_names['another'] ],
            [ 'assertNotSame', ProductGroup::GROUP_AP, self::$product_names['another'] ],
            [ 'assertNotSame', ProductGroup::GROUP_D, self::$product_names['another'] ],
            [ 'assertNotSame', ProductGroup::GROUP_L, self::$product_names['another'] ],
        ];
    }


    /**
     * @dataProvider dataTestProductGroupNameDetection
     */
    public function testProductGroupNameDetection($assert_method, $expected_group_name, $product_name)
    {
        $product = new ShopItem();

        $product->name = $product_name;
        $pg = new ProductGroup($product);

        $detected_group_name = $pg->getName();

        $this->{$assert_method}($detected_group_name, $expected_group_name);
    }



    public static function dataTestProductMinQtyCalculations()
    {
        return [
            'atov_5_1'  =>  [ self::$product_names['atov'],  5, 1,  0 ],
            'atov_30_1' =>  [ self::$product_names['atov'], 30, 1,  0 ],
            'atov_65_1' =>  [ self::$product_names['atov'], 65, 1, 10 ],

            'atov_5_10'  => [ self::$product_names['atov'],  5, 10,  0 ],
            'atov_30_10' => [ self::$product_names['atov'], 30, 10,  0 ],
            'atov_65_10' => [ self::$product_names['atov'], 65, 10, 19 ],

            'atov_5_31'  => [ self::$product_names['atov'],  5, 31,  0 ],
            'atov_30_31' => [ self::$product_names['atov'], 30, 31,  0 ],
            'atov_65_31' => [ self::$product_names['atov'], 65, 31, 40 ],


            'malarone_5_1'  =>  [ self::$product_names['malarone'],  5, 1,  0 ],
            'malarone_30_1' =>  [ self::$product_names['malarone'], 30, 1,  0 ],
            'malarone_65_1' =>  [ self::$product_names['malarone'], 65, 1, 10 ],

            'malarone_5_10'  => [ self::$product_names['malarone'],  5, 10,  0 ],
            'malarone_30_10' => [ self::$product_names['malarone'], 30, 10,  0 ],
            'malarone_65_10' => [ self::$product_names['malarone'], 65, 10, 19 ],

            'malarone_5_31'  => [ self::$product_names['malarone'],  5, 31,  0 ],
            'malarone_30_31' => [ self::$product_names['malarone'], 30, 31,  0 ],
            'malarone_65_31' => [ self::$product_names['malarone'], 65, 31, 40 ],


            'maloff_5_1'  =>  [ self::$product_names['maloff'],  5, 1,  0 ],
            'maloff_30_1' =>  [ self::$product_names['maloff'], 30, 1,  0 ],
            'maloff_65_1' =>  [ self::$product_names['maloff'], 65, 1, 10 ],

            'maloff_5_10'  => [ self::$product_names['maloff'],  5, 10,  0 ],
            'maloff_30_10' => [ self::$product_names['maloff'], 30, 10,  0 ],
            'maloff_65_10' => [ self::$product_names['maloff'], 65, 10, 19 ],

            'maloff_5_31'  => [ self::$product_names['maloff'],  5, 31,  0 ],
            'maloff_30_31' => [ self::$product_names['maloff'], 30, 31,  0 ],
            'maloff_65_31' => [ self::$product_names['maloff'], 65, 31, 40 ],


            'atov_p_5_1'  => [ self::$product_names['atov_p'],  5, 1,  0 ],
            'atov_p_10_1' => [ self::$product_names['atov_p'], 10, 1, 10 ],
            'atov_p_15_1' => [ self::$product_names['atov_p'], 15, 1, 10 ],
            'atov_p_20_1' => [ self::$product_names['atov_p'], 20, 1, 20 ],
            'atov_p_25_1' => [ self::$product_names['atov_p'], 25, 1, 20 ],
            'atov_p_30_1' => [ self::$product_names['atov_p'], 30, 1, 30 ],
            'atov_p_35_1' => [ self::$product_names['atov_p'], 35, 1, 30 ],
            'atov_p_40_1' => [ self::$product_names['atov_p'], 40, 1,  0 ],
            'atov_p_65_1' => [ self::$product_names['atov_p'], 65, 1,  0 ],

            'atov_p_5_10'  => [ self::$product_names['atov_p'],  5, 10,  0 ],
            'atov_p_10_10' => [ self::$product_names['atov_p'], 10, 10, 19 ],
            'atov_p_15_10' => [ self::$product_names['atov_p'], 15, 10, 19 ],
            'atov_p_20_10' => [ self::$product_names['atov_p'], 20, 10, 38 ],
            'atov_p_25_10' => [ self::$product_names['atov_p'], 25, 10, 38 ],
            'atov_p_30_10' => [ self::$product_names['atov_p'], 30, 10, 57 ],
            'atov_p_35_10' => [ self::$product_names['atov_p'], 35, 10, 57 ],
            'atov_p_40_10' => [ self::$product_names['atov_p'], 40, 10,  0 ],
            'atov_p_65_10' => [ self::$product_names['atov_p'], 65, 10,  0 ],

            'atov_p_5_31'  => [ self::$product_names['atov_p'],  5, 31,  0 ],
            'atov_p_10_31' => [ self::$product_names['atov_p'], 10, 31, 40 ],
            'atov_p_15_31' => [ self::$product_names['atov_p'], 15, 31, 40 ],
            'atov_p_20_31' => [ self::$product_names['atov_p'], 20, 31, 80 ],
            'atov_p_25_31' => [ self::$product_names['atov_p'], 25, 31, 80 ],
            'atov_p_30_31' => [ self::$product_names['atov_p'], 30, 31, 120 ],
            'atov_p_35_31' => [ self::$product_names['atov_p'], 35, 31, 120 ],
            'atov_p_40_31' => [ self::$product_names['atov_p'], 40, 31,  0 ],
            'atov_p_65_31' => [ self::$product_names['atov_p'], 65, 31,  0 ],


            'malarone_p_5_1'  => [ self::$product_names['malarone_p'],  5, 1,  0 ],
            'malarone_p_10_1' => [ self::$product_names['malarone_p'], 10, 1, 10 ],
            'malarone_p_15_1' => [ self::$product_names['malarone_p'], 15, 1, 10 ],
            'malarone_p_20_1' => [ self::$product_names['malarone_p'], 20, 1, 20 ],
            'malarone_p_25_1' => [ self::$product_names['malarone_p'], 25, 1, 20 ],
            'malarone_p_30_1' => [ self::$product_names['malarone_p'], 30, 1, 30 ],
            'malarone_p_35_1' => [ self::$product_names['malarone_p'], 35, 1, 30 ],
            'malarone_p_40_1' => [ self::$product_names['malarone_p'], 40, 1,  0 ],
            'malarone_p_65_1' => [ self::$product_names['malarone_p'], 65, 1,  0 ],

            'malarone_p_5_10'  => [ self::$product_names['malarone_p'],  5, 10,  0 ],
            'malarone_p_10_10' => [ self::$product_names['malarone_p'], 10, 10, 19 ],
            'malarone_p_15_10' => [ self::$product_names['malarone_p'], 15, 10, 19 ],
            'malarone_p_20_10' => [ self::$product_names['malarone_p'], 20, 10, 38 ],
            'malarone_p_25_10' => [ self::$product_names['malarone_p'], 25, 10, 38 ],
            'malarone_p_30_10' => [ self::$product_names['malarone_p'], 30, 10, 57 ],
            'malarone_p_35_10' => [ self::$product_names['malarone_p'], 35, 10, 57 ],
            'malarone_p_40_10' => [ self::$product_names['malarone_p'], 40, 10,  0 ],
            'malarone_p_65_10' => [ self::$product_names['malarone_p'], 65, 10,  0 ],

            'malarone_p_5_31'  => [ self::$product_names['malarone_p'],  5, 31,  0 ],
            'malarone_p_10_31' => [ self::$product_names['malarone_p'], 10, 31, 40 ],
            'malarone_p_15_31' => [ self::$product_names['malarone_p'], 15, 31, 40 ],
            'malarone_p_20_31' => [ self::$product_names['malarone_p'], 20, 31, 80 ],
            'malarone_p_25_31' => [ self::$product_names['malarone_p'], 25, 31, 80 ],
            'malarone_p_30_31' => [ self::$product_names['malarone_p'], 30, 31, 120 ],
            'malarone_p_35_31' => [ self::$product_names['malarone_p'], 35, 31, 120 ],
            'malarone_p_40_31' => [ self::$product_names['malarone_p'], 40, 31,  0 ],
            'malarone_p_65_31' => [ self::$product_names['malarone_p'], 65, 31,  0 ],


            'doxy_5_1'  =>  [ self::$product_names['doxy'],  5, 1, 31 ],
            'doxy_30_1' =>  [ self::$product_names['doxy'], 30, 1, 31 ],
            'doxy_65_1' =>  [ self::$product_names['doxy'], 65, 1, 31 ],

            'doxy_5_10'  => [ self::$product_names['doxy'],  5, 10, 40 ],
            'doxy_30_10' => [ self::$product_names['doxy'], 30, 10, 40 ],
            'doxy_65_10' => [ self::$product_names['doxy'], 65, 10, 40 ],

            'doxy_5_31'  => [ self::$product_names['doxy'],  5, 31, 61 ],
            'doxy_30_31' => [ self::$product_names['doxy'], 30, 31, 61 ],
            'doxy_65_31' => [ self::$product_names['doxy'], 65, 31, 61 ],


            'lariam_5_1'  =>  [ self::$product_names['lariam'],  5, 1, 0 ],
            'lariam_10_1' =>  [ self::$product_names['lariam'], 10, 1, 2 ],
            'lariam_20_1' =>  [ self::$product_names['lariam'], 20, 1, 3 ],
            'lariam_30_1' =>  [ self::$product_names['lariam'], 30, 1, 5 ],
            'lariam_65_1' =>  [ self::$product_names['lariam'], 65, 1, 6 ],

            'lariam_5_10'  => [ self::$product_names['lariam'],  5, 10, 0 ],
            'lariam_10_10' => [ self::$product_names['lariam'], 10, 10, 2 ],
            'lariam_20_10' => [ self::$product_names['lariam'], 20, 10, 4 ],
            'lariam_30_10' => [ self::$product_names['lariam'], 30, 10, 6 ],
            'lariam_65_10' => [ self::$product_names['lariam'], 65, 10, 7 ],

            'lariam_5_31'  => [ self::$product_names['lariam'],  5, 31, 0 ],
            'lariam_10_31' => [ self::$product_names['lariam'], 10, 31, 3 ],
            'lariam_20_31' => [ self::$product_names['lariam'], 20, 31, 5 ],
            'lariam_30_31' => [ self::$product_names['lariam'], 30, 31, 8 ],
            'lariam_65_31' => [ self::$product_names['lariam'], 65, 31, 10 ],


            'another_5_1'  =>  [ self::$product_names['another'],  5, 1, 1 ],
            'another_30_1' =>  [ self::$product_names['another'], 30, 1, 1 ],
            'another_65_1' =>  [ self::$product_names['another'], 65, 1, 1 ],

            'another_5_10'  => [ self::$product_names['another'],  5, 10, 1 ],
            'another_30_10' => [ self::$product_names['another'], 30, 10, 1 ],
            'another_65_10' => [ self::$product_names['another'], 65, 10, 1 ],

            'another_5_31'  => [ self::$product_names['another'],  5, 31, 1 ],
            'another_30_31' => [ self::$product_names['another'], 30, 31, 1 ],
            'another_65_31' => [ self::$product_names['another'], 65, 31, 1 ],
        ];
    }


    /**
     * @dataProvider dataTestProductMinQtyCalculations
     */
    public function testProductMinimumQuantityCalculations($product_name, $weight, $days_away, $expected_qty)
    {
        $product = new ShopItem();

        $product->name = $product_name;
        $pg = new ProductGroup($product);

        $calculated_qty = ShopHelper::getMinimumQuantity($product, $weight, $days_away);

        $this->assertSame($calculated_qty, $expected_qty);
    }



}
