<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {

        Schema::create('admin_menu', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('sub_id')->index('sub_id');
            $table->string('title');
            $table->string('path');
            $table->string('activePath');
            $table->string('constant');
            $table->string('icon');
            $table->integer('new_window');
            $table->integer('order')->index('order');
        });

        Schema::create('admin_permissions', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->string('owner_type');
            $table->string('owner_id');
            $table->timestamps();
        });

        Schema::create('admin_role_permissions', function (Blueprint $table) {
            $table->integer('role_id');
            $table->integer('permission_id');
            $table->timestamps();

            $table->unique(['role_id', 'permission_id']);
        });

        Schema::create('admin_roles', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->timestamps();
        });

        Schema::create('admin_user_allowed_categories', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('category_id')->index('admin_user_restricted_categories_category_id_index');
            $table->integer('user_id')->index('admin_user_restricted_categories_user_id_index');
            $table->timestamps();
        });

        Schema::create('admin_user_policy_confirmations', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedBigInteger('user_id')->index();
            $table->unsignedBigInteger('policy_id')->index();
            $table->timestamps();
        });

        Schema::create('admin_users', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('username')->index('username');
            $table->string('password')->index('password');
            $table->string('email')->index('email');
            $table->string('signature')->nullable();
            $table->string('session_id')->nullable();
            $table->string('name');
            $table->integer('role')->index()->default(0);
            $table->text('credentials')->nullable();
            $table->dateTime('last_login')->index('last_login')->nullable();
            $table->timestamp('last_updated')->useCurrentOnUpdate()->useCurrent();
        });

        Schema::create('admin_users__favourite_links', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('user_id');
            $table->integer('menuitem_id')->index('admin_users__favourite_links_menuitem_id_foreign');
            $table->timestamps();

            $table->unique(['user_id', 'menuitem_id'], 'admin_users__favourite_links__uniqu_user_menuitem');
        });

        Schema::create('admin_users__linked_users', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('user_id');
            $table->integer('linked_user_id')->index('admin_users__linked_users_linked_user_id_foreign');
            $table->timestamps();

            $table->unique(['user_id', 'linked_user_id'], 'admin_users__linked_users__uniqu_user_user');
        });

        Schema::create('approval_log', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('user_id')->index();
            $table->integer('order_item_id')->index();
            $table->string('status')->default('');
            $table->timestamps();
            $table->string('ip_address')->nullable();
            $table->string('browser', 250)->nullable();
        });

        Schema::create('assessment_answers', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('magento_answer_id')->nullable();
            $table->integer('assessment_id');
            $table->integer('form_id');
            $table->integer('question_id');
            $table->text('answer')->nullable();
            $table->timestamps();
            $table->integer('answer_change_count')->default(0);
            $table->boolean('answer_confirmed')->nullable();
            $table->boolean('answer_changed_after_confirmation')->nullable();

            $table->index(['assessment_id', 'form_id', 'question_id']);
        });

        Schema::create('assessment_answers_magento', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('mtc_order_id')->index('assessment_answers_magento_mtc_order_id_foreign');
            $table->unsignedInteger('magento_order_id');
            $table->unsignedInteger('magento_answer_id')->index();
            $table->text('magento_answer');
            $table->dateTime('magento_creation_time');
            $table->dateTime('magento_update_time');
        });

        Schema::create('assessment_forms', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->tinyInteger('is_active')->default(0);
            $table->unsignedInteger('icon_id')->nullable();
            $table->timestamps();
        });

        Schema::create('assessment_qualifying_answers', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('question_id');
            $table->string('qualifying_answer');
            $table->unsignedInteger('group_id')->nullable()->index('assessment_qa__group_id');
            $table->timestamps();

            $table->unique(['question_id', 'group_id'], 'assessment_qa__unique_question_id_group_id');
        });

        Schema::create('assessment_qualifying_answers__group_templates', function (Blueprint $table) {
            $table->increments('id');
            $table->char('label', 1);
            $table->char('hex_rgb', 9)->default('#eeeeee');
            $table->timestamps();
        });

        Schema::create('assessment_qualifying_answers__groups', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('template_id')->default(1);
            $table->enum('logic', ['and', 'or'])->default('and');
            $table->unsignedInteger('config_id')->nullable()->index('assessment_qa__groups__config_id');
            $table->timestamps();

            $table->unique(['template_id', 'config_id'], 'assessment_qa__groups__unique_t_c');
        });

        Schema::create('assessment_qualifying_answers__groups_config', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->unique('assessment_qa__groups_config__unique_item_id');
            $table->enum('logic', ['and', 'or'])->default('and');
            $table->timestamps();
        });

        Schema::create('assessment_questions', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('magento_question_id')->nullable();
            $table->integer('old_id');
            $table->integer('form_id');
            $table->string('question_type', 50);
            $table->text('question_text');
            $table->text('variations');
            $table->string('clearall_answer')->nullable();
            $table->tinyInteger('is_multiple')->default(0);
            $table->text('block_if');
            $table->text('alert_if');
            $table->text('okay_if');
            $table->text('cant_proceed_if');
            $table->integer('show_if_question');
            $table->text('show_if_answer');
            $table->integer('answer_expires_after_days')->default(0);
            $table->tinyInteger('is_required')->default(0);
            $table->tinyInteger('is_active')->default(1);
            $table->integer('sort');
            $table->timestamps();
            $table->dateTime('deleted_at')->nullable();
            $table->text('settings');

            $table->index(['old_id', 'form_id']);
        });

        Schema::create('assessments', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('member_id')->index('member_id');
            $table->integer('form_id')->index('form_id');
            $table->string('customer_name', 100)->nullable();
            $table->date('customer_dob')->nullable();
            $table->integer('category_id')->default(0);
            $table->integer('item_id');
            $table->tinyInteger('is_completed')->default(0)->index('is_completed');
            $table->text('additional_info')->nullable();
            $table->timestamps();
        });

        Schema::create('asset_gallery__icons', function (Blueprint $table) {
            $table->increments('id');
            $table->string('filepath');
            $table->string('label');
            $table->timestamps();
        });

        Schema::create('audit_log', function (Blueprint $table) {
            $table->comment('Used to monitor changes on tables. See the settings table for triggers currently available.');
            $table->increments('id');
            $table->unsignedInteger('subject_id')->index('subject_id');
            $table->string('subject_type', 50)->default('')->index('subject_type');
            $table->string('name')->default('');
            $table->binary('data')->nullable();
            $table->timestamp('date')->useCurrent();
        });

        Schema::create('basket', function (Blueprint $table) {
            $table->integer('id', true);
            $table->unsignedInteger('subscription_id')->nullable()->index('basket__subscription_id');
            $table->dateTime('date')->useCurrent();
            $table->integer('delivery')->default(0);
            $table->boolean('billing_address_used')->default(true);
            $table->string('coupon')->default('')->index('coupon');
            $table->integer('member')->default(0)->index('member');
            $table->string('ref');
            $table->string('keywords');
            $table->timestamp('timestamp')->useCurrent();
            $table->dateTime('basket_expiry_time');
            $table->string('checkout', 6);
            $table->string('seccode', 50);
            $table->integer('flag_type');
            $table->text('delivery_instructions')->nullable();
        });

        Schema::create('basket_address', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('basket_id')->default(0)->index('basket_id');
            $table->set('type', ['billing', 'shipping'])->default('')->index('type');
            $table->string('gender', 10);
            $table->string('title', 10);
            $table->string('firstname')->default('');
            $table->string('middle_name')->nullable();
            $table->string('lastname')->default('');
            $table->string('address1')->default('');
            $table->string('address2')->default('');
            $table->string('city')->default('');
            $table->string('country')->default('');
            $table->string('postcode')->default('');
            $table->string('state');
            $table->timestamp('timestamp')->useCurrent();
            $table->string('notes')->nullable();
        });

        Schema::create('basket_info', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('basket_id')->default(0)->index('basket_id');
            $table->string('email')->nullable()->default('');
            $table->string('contact_no')->default('');
            $table->text('message');
            $table->integer('newsletter')->default(1);
            $table->string('how_did_you_hear_about_us')->default('');
            $table->string('how_did_you_hear_about_us_text')->default('');
            $table->timestamp('timestamp')->useCurrent();
            $table->string('phone_prefix');
            $table->smallInteger('dob_date')->nullable();
            $table->smallInteger('dob_month')->nullable();
            $table->smallInteger('dob_year')->nullable();
            $table->longText('assessment_ids')->nullable();
            $table->unsignedInteger('multisite__site_id')->nullable()->index('basket_info_multisite__site_id_foreign');
        });

        Schema::create('basket_items', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('basket_id')->default(0)->index('basket_id');
            $table->integer('item_id')->default(0)->index('item_id');
            $table->unsignedInteger('bundle_id')->nullable()->index('basket_items__bundle_id');
            $table->unsignedInteger('prescription_item_id')->nullable()->index('basket_items_prescription_item_id_foreign');
            $table->boolean('has_requested_subscription')->default(false);
            $table->integer('assessment_id')->default(0);
            $table->integer('quantity')->default(0);
            $table->integer('recommended_quantity')->default(0);
            $table->string('size', 50)->default('');
            $table->string('PLU', 50);
            $table->integer('sizeid')->index('sizeid');
            $table->boolean('nhs_prescription')->default(false);
            $table->date('refill_date');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('basket_protx', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('basket_id')->default(0)->index('basket_id');
            $table->double('price')->default(0);
            $table->dateTime('date')->useCurrent();
            $table->string('transaction_type', 50)->default('payment');
            $table->string('secKey', 50);
            $table->string('VPSTxId', 50);
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('basket_recovery', function (Blueprint $table) {
            $table->increments('id');
            $table->string('type', 10)->index('type');
            $table->string('interval');
            $table->unsignedInteger('interval_seconds');
            $table->string('name');
            $table->string('subject');
            $table->text('content');
            $table->string('sms_content');
            $table->boolean('is_button_visible')->default(true);
            $table->string('button_url')->nullable();
            $table->string('button_copy')->nullable();
            $table->text('coupon_data')->nullable();
            $table->boolean('is_active')->index('is_active');
            $table->timestamps();
            $table->tinyInteger('once_per_customer');
            $table->integer('restriction_id');
            $table->string('restriction_type');
            $table->string('trigger');

            $table->index(['restriction_id', 'restriction_type']);
        });

        Schema::create('basket_recovery_logs', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('basket_recovery_id')->index('basket_recovery_logs_basket_recovery_id_foreign');
            $table->unsignedInteger('basket_id')->index();
            $table->string('action')->index();
            $table->string('value')->nullable();
            $table->timestamps();
        });

        Schema::create('basket_recovery_sent_emails', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('basket_id')->index();
            $table->unsignedInteger('recovery_id')->index();
            $table->timestamps();
        });

        Schema::create('black_list', function (Blueprint $table) {
            $table->increments('id');
            $table->tinyInteger('active')->default(1)->index('active');
            $table->integer('order')->default(0)->index('order');
            $table->string('telephone', 50);
            $table->string('email');
            $table->text('address');
            $table->tinyInteger('hard_blacklist');
        });

        Schema::create('blog_categories', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->index('name');
            $table->tinyInteger('active')->default(1)->index('active');
            $table->integer('order')->default(0)->index('order');
        });

        Schema::create('brands', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order')->index('order');
            $table->string('name')->default('')->index('name');
            $table->string('image')->default('');
            $table->string('logo')->default('');
            $table->text('description');
            $table->boolean('hide')->default(false)->index('hide');
            $table->boolean('deleted')->index('deleted');
            $table->integer('stock')->default(0);
            $table->text('seo_title');
            $table->text('seo_keywords');
            $table->text('seo_description');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('bundle_items', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('bundle_id')->index('bundle_items_bundle_id_foreign');
            $table->unsignedInteger('bundle_type_item_id')->nullable()->index('bundle_items__bundle_type_item_id');
            $table->timestamps();
        });

        Schema::create('bundle_types', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->string('url');
            $table->unsignedInteger('assessment_form_id')->nullable()->index('bundle_types__assessment_form_id');
        });

        Schema::create('bundle_types_items', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('bundle_type_id')->index('bundle_types_items_bundle_type_id_foreign');
            $table->integer('shop_item_id');
            $table->integer('shop_item_size_id')->nullable()->index('bundle_types_items__shop_item_size_id');
            $table->timestamps();

            $table->unique(['shop_item_id', 'shop_item_size_id']);
        });

        Schema::create('bundles', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('type_id')->index('bundles_type_id_foreign');
            $table->string('name');
            $table->timestamps();
        });

        Schema::create('catalog_import', function (Blueprint $table) {
            $table->increments('id');
            $table->string('nodeable_type');
            $table->unsignedBigInteger('nodeable_id');
            $table->string('ref');
            $table->timestamps();

            $table->index(['nodeable_type', 'nodeable_id']);
        });

        Schema::create('categories', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('form_id')->index('form_id');
            $table->integer('sub_id')->nullable()->index('sub_id');
            $table->string('name')->default('')->index('name');
            $table->string('display_name')->nullable();
            $table->string('old_slug');
            $table->text('slug');
            $table->integer('order')->default(0)->index('order');
            $table->string('background_colour', 9)->nullable();
            $table->boolean('hide')->default(false)->index('hide');
            $table->string('description_title')->nullable();
            $table->text('description');
            $table->string('description_image')->nullable();
            $table->string('description_image_alt')->nullable();
            $table->text('information_leaflet');
            $table->text('side_effects');
            $table->text('cautions');
            $table->string('overview_title')->nullable();
            $table->text('overview');
            $table->string('causes_title')->nullable();
            $table->text('causes');
            $table->string('symptoms_title')->nullable();
            $table->text('symptoms');
            $table->string('treatments_title')->nullable();
            $table->text('treatments');
            $table->string('faq_title')->nullable();
            $table->text('faq');
            $table->string('faq_image')->nullable();
            $table->string('faq_image_alt')->nullable();
            $table->integer('deleted')->default(0)->index('deleted');
            $table->string('image');
            $table->string('url_to_category_products', 256)->nullable();
            $table->text('seo_title');
            $table->text('seo_keywords');
            $table->text('seo_description');
            $table->timestamp('timestamp')->useCurrent();
            $table->integer('custom_field_set_id')->index('custom_field_set_id');
            $table->dateTime('require_id_checks')->nullable()->index();
            $table->text('faq_cache');
            $table->boolean('is_online_doctor')->default(false);
            $table->boolean('hide_pharmacy_items');
            $table->integer('lft')->index();
            $table->integer('rgt')->index();
            $table->integer('depth')->index();
            $table->string('image_alt')->nullable()->index();
            $table->double('starting_price')->nullable();
            $table->text('description_small')->nullable();
            $table->string('product_list_title')->nullable();
            $table->text('product_list_subtitle')->nullable();
            $table->string('content_blocks_title')->nullable();
            $table->text('content_blocks_subtitle')->nullable();
            $table->string('content_block_1_id')->nullable();
            $table->string('content_block_2_id')->nullable();
            $table->string('content_block_3_id')->nullable();

            $table->fullText(['name'], 'name_2');
        });

        Schema::create('categories__display_tags', function (Blueprint $table) {
            $table->increments('id');
            $table->char('group_name', 1)->default('a');
            $table->integer('category_id')->index('categories__display_tags_category_id_foreign');
            $table->string('label');
            $table->unsignedInteger('icon_id')->nullable()->index('categories__dt__icon_id');
            $table->timestamps();
        });

        Schema::create('categories__info_sections', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('category_id')->index('categories__info_sections_category_id_foreign');
            $table->string('title')->nullable();
            $table->boolean('is_content_itemised')->default(false);
            $table->text('content')->nullable();
            $table->integer('order')->default(0);
            $table->timestamps();
        });

        Schema::create('category_item_comparison', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('category_id')->index();
            $table->string('item_name');
            $table->text('feature_name');
            $table->text('value');
            $table->timestamps();
        });

        Schema::create('category_restricted_zones', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('category_id')->index();
            $table->integer('zone');
            $table->timestamps();
        });

        Schema::create('countries', function (Blueprint $table) {
            $table->integer('id', true);
            $table->boolean('status')->default(true)->index('status');
            $table->integer('order')->default(0)->index('order');
            $table->char('code', 2)->default('')->index('code');
            $table->string('name', 45)->default('')->index('name');
            $table->char('currency_code', 3)->nullable()->index('currency_code');
            $table->double('north')->nullable()->index('north');
            $table->double('south')->nullable()->index('south');
            $table->double('east')->nullable()->index('east');
            $table->double('west')->nullable()->index('west');
            $table->string('capital_name', 30)->nullable();
            $table->string('continent_name', 15)->nullable();
            $table->char('continent_code', 2)->nullable()->index('continent_code');
            $table->string('languages', 100)->nullable();
            $table->char('iso_code', 3)->nullable()->index('iso_code');
            $table->string('dial_code', 10)->nullable()->index('dial_code');
            $table->boolean('is_eu')->default(false)->index('is_eu');
            $table->boolean('has_postcodes')->default(true)->index('has_postcodes');
            $table->string('regex', 50)->nullable()->index('regex');
        });

        Schema::create('country_states', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('country_id')->default(0)->index('country_id');
            $table->string('name', 31)->index('name');
            $table->string('code', 2)->index('code');
            $table->boolean('status')->default(true)->index('status');
        });

        Schema::create('coupons', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('code')->default('')->unique('code');
            $table->date('from_date')->default('0000-00-00')->index('from_date');
            $table->date('exp_date')->default('0000-00-00')->index('exp_date');
            $table->integer('redemptions')->default(0)->index('redemptions');
            $table->set('type', ['freedel', 'set', 'percent', 'xfory', 'xgety'])->default('')->index('type');
            $table->enum('structure', ['single', 'multi'])->default('single')->index('structure')->comment('Coupon structure. Multi code coupons have codes in coupons_codes table');
            $table->double('value')->default(0)->index('value');
            $table->string('name')->default('');
            $table->string('code_type', 50)->default('coupon')->index('code_type');
            $table->boolean('non_sale_only')->index('non_sale_only');
            $table->tinyInteger('free_delivery')->index('free_delivery')->comment('0=no,1=all,2=uk only');
            $table->double('min_price')->index('min_price');
            $table->boolean('first_order')->index('first_order');
            $table->dateTime('created_at');
            $table->dateTime('updated_at');
        });

        Schema::create('coupons_codes', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('coupon_id')->index('coupon_id');
            $table->string('code', 50)->unique('code');
            $table->integer('order_id')->default(0)->index('order_id');
            $table->dateTime('created_at');
            $table->dateTime('updated_at');
        });

        Schema::create('coupons_conditions', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('coupon_id');
            $table->set('type', ['item', 'brand', 'category'])->index('type');
            $table->boolean('exclude')->index('exclude');
            $table->integer('value');
            $table->dateTime('created_at');
            $table->dateTime('updated_at');

            $table->index(['coupon_id', 'type'], 'coupon_id_type');
        });

        Schema::create('currencies', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('currency')->default('')->index('currency');
            $table->double('ratio')->default(0)->index('ratio');
            $table->timestamp('timestamp')->useCurrent();
            $table->tinyInteger('enabled');
            $table->string('symbol');
            $table->string('name');
            $table->string('country_code');
        });

        Schema::create('custom_fields', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('name');
            $table->string('type');
            $table->integer('custom_field_list_id');
            $table->string('database_field');
            $table->boolean('multiple');
            $table->integer('quantity');
            $table->integer('sort');
            $table->dateTime('created_at');
            $table->dateTime('updated_at');
        });

        Schema::create('custom_fields_sets', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('name');
            $table->tinyInteger('default')->index('default');
            $table->dateTime('created_at');
            $table->dateTime('updated_at');
        });

        Schema::create('custom_fields_sets_fields', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('set_id')->index('set_id');
            $table->integer('custom_field_id')->index('custom_field_id');
            $table->tinyInteger('required');
            $table->tinyInteger('items');
            $table->tinyInteger('variations');
            $table->tinyInteger('filter');
            $table->integer('order')->index('order');
            $table->dateTime('created_at');
            $table->dateTime('updated_at');
        });

        Schema::create('customers_also_viewed', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->index();
            $table->integer('also_viewed_id')->index();
            $table->integer('views')->index();
            $table->timestamps();
        });

        Schema::create('delivery_couriers_queue', function (Blueprint $table) {
            $table->increments('id');
            $table->text('order_object');
            $table->timestamp('created_at')->useCurrent();
            $table->timestamp('processed_at')->nullable();
            $table->string('response', 16)->nullable();
            $table->enum('status', ['pending', 'processing', 'success', 'failed'])->default('pending');
        });

        Schema::create('delivery_method_rates', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('method_id')->index();
            $table->decimal('weight_min')->index();
            $table->decimal('weight_max')->index();
            $table->decimal('line_cost')->nullable();
            $table->decimal('kg_cost')->nullable();
            $table->timestamps();
        });

        Schema::create('delivery_methods', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('name')->default('');
            $table->double('cost')->default(0);
            $table->double('item_combined_cost')->default(0);
            $table->double('min')->default(0);
            $table->double('max')->default(0);
            $table->boolean('heavy')->default(false);
            $table->integer('order');
            $table->integer('zone')->index('zone');
            $table->integer('subzone');
            $table->integer('exzone');
            $table->tinyInteger('disable');
            $table->timestamp('timestamp')->useCurrent();
            $table->integer('service_id')->index();
        });

        Schema::create('delivery_services_options', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('service_id')->index('service_id');
            $table->string('name')->index('name');
            $table->string('code', 55);
            $table->string('type', 20);
            $table->string('format', 50);
            $table->boolean('signed_for')->default(false);
            $table->string('service_enhancement', 50);
            $table->dateTime('created_at');
            $table->timestamp('updated_at')->useCurrentOnUpdate()->useCurrent();

            $table->index(['id'], 'id');
        });

        Schema::create('delivery_zones', function (Blueprint $table) {
            $table->string('country_code', 2)->default('');
            $table->string('country_name')->default('');
            $table->integer('zone');
            $table->tinyInteger('disable');

            $table->primary(['country_code', 'country_name', 'zone']);
        });

        Schema::create('discountoffers', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('name');
            $table->string('type');
            $table->dateTime('start_date');
            $table->dateTime('end_date');
            $table->decimal('value1', 10);
            $table->decimal('value2', 10);
            $table->text('restrictions_brand');
            $table->text('restrictions_band');
            $table->text('restrictions_categories');
            $table->text('restrictions_items');
            $table->text('restrictions_campaigns');
            $table->double('restrictions_min_value', 10, 2);
            $table->dateTime('date_added');
            $table->integer('redemptions');
            $table->tinyInteger('disabled');
            $table->integer('freeitem');
            $table->tinyInteger('firstorder');
            $table->tinyInteger('allstock');
        });

        Schema::create('doctor_policies', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->index('name');
            $table->tinyInteger('active')->default(1)->index('active');
            $table->integer('order')->default(0)->index('order');
            $table->text('url');
        });

        Schema::create('doctor_surgeries', function (Blueprint $table) {
            $table->increments('id');
            $table->string('area_team_name');
            $table->string('ccg_name');
            $table->string('ods_code');
            $table->string('practice_name');
            $table->string('address_1');
            $table->string('address_2');
            $table->string('address_3')->nullable();
            $table->string('city')->nullable();
            $table->string('county')->nullable();
            $table->string('phone')->nullable();
            $table->string('email')->nullable();
            $table->string('fax')->nullable();
            $table->string('website')->nullable();
            $table->string('postcode');
            $table->string('latitude', 50)->index('latitude');
            $table->string('longitude', 50)->index('longitude');
            $table->string('status');
            $table->string('planned');
            $table->string('go_live');
            $table->string('patient_list_size');
            $table->string('percent_estimate');
            $table->string('epsr2_estimate');
            $table->timestamps();
            $table->softDeletes();
        });

        Schema::create('emails_double_optin', function (Blueprint $table) {
            $table->increments('id');
            $table->string('email');
            $table->string('firstname');
            $table->string('surname');
            $table->timestamps();

            $table->index(['id']);
        });

        Schema::create('emails_lists', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('name');
        });

        Schema::create('emails_lists_mails', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('list_id')->index('list_id');
            $table->string('email')->index('email');
            $table->string('firstname');
            $table->string('surname');
            $table->string('title', 32);
        });

        Schema::create('emails_queued', function (Blueprint $table) {
            $table->increments('id');
            $table->string('email');
            $table->text('subject');
            $table->string('from_email');
            $table->string('from_name');
            $table->longText('content');
            $table->tinyInteger('dev_copy')->default(0);
            $table->dateTime('sent_at')->nullable();
            $table->timestamps();
        });

        Schema::create('evopay_responses', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')->index();
            $table->string('status');
            $table->text('response');
            $table->timestamps();
        });

        Schema::create('featured_products', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->index();
            $table->string('sex');
            $table->smallInteger('order')->index();
            $table->timestamps();
        });

        Schema::create('feefo_reviews_logs', function (Blueprint $table) {
            $table->increments('id');
            $table->text('request');
            $table->text('response');
            $table->timestamps();
        });

        Schema::create('feefo_reviews_settings', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->string('url');
            $table->string('telephone');
            $table->string('streetAddress');
            $table->string('addressLocality');
            $table->string('addressLocality2');
            $table->string('addressCountry');
            $table->timestamps();
        });

        Schema::create('forms', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('published')->index('published');
            $table->string('name');
            $table->integer('save_db');
            $table->integer('send_email');
            $table->string('email_receiver');
            $table->string('redirect');
            $table->integer('no_limit');
            $table->dateTime('display_until');
            $table->integer('type');
            $table->string('css_id');
            $table->string('css_class');
            $table->string('button_class');
            $table->string('submit_text');
            $table->integer('callback');
            $table->string('callback_class');
            $table->string('callback_action');
            $table->text('info_text');
            $table->text('success_text');
        });

        Schema::create('forms_fieldsets', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('form_id')->index('form_id');
            $table->string('name');
            $table->string('css_class');
            $table->string('css_id');
            $table->tinyInteger('show_title');
            $table->string('placeholder');
            $table->integer('order');
        });

        Schema::create('forms_questions', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('form_id')->index('form_id');
            $table->integer('fieldset_id')->index('fieldset_id');
            $table->string('name');
            $table->string('slug');
            $table->string('placeholder');
            $table->string('css_class');
            $table->string('type');
            $table->string('validate');
            $table->string('regex');
            $table->integer('required');
            $table->integer('freetext_answer');
            $table->integer('order');
        });

        Schema::create('forms_questions_emails', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('question_id')->index('question_id');
            $table->string('value');
        });

        Schema::create('forms_questions_options', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('question_id')->index('question_id');
            $table->string('value');
            $table->integer('order');
            $table->integer('correct');
        });

        Schema::create('forms_responses', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('form_id')->index('form_id');
            $table->dateTime('date');
            $table->string('ip');
        });

        Schema::create('forms_responses_answers', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('response_id')->index('response_id');
            $table->integer('question_id')->index('question_id');
            $table->string('value');
            $table->integer('value_id')->index('value_id');
        });

        Schema::create('incompatibilities', function (Blueprint $table) {
            $table->increments('id');
            $table->text('message');
            $table->timestamps();
        });

        Schema::create('incompatible_categories', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('incompatibility_id');
            $table->integer('category_id');
            $table->timestamps();
        });

        Schema::create('incompatible_medications', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->index();
            $table->string('level')->nullable();
            $table->string('medication')->nullable();
            $table->string('message')->nullable();
            $table->timestamps();
        });

        Schema::create('ingredients', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->integer('limit_period');
            $table->integer('limit_amount');
            $table->integer('limit_order');
            $table->integer('limit_once');
            $table->string('limit_type', 20);
            $table->text('message');
            $table->timestamps();
        });

        Schema::create('inventories', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('sku')->index();
            $table->string('warehouse_code')->index();
            $table->bigInteger('quantity')->nullable()->default(0);
            $table->timestamps();

            $table->unique(['sku', 'warehouse_code']);
        });

        Schema::create('item_delivery_rates', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('item_id')->index('item_id');
            $table->integer('delivery_method_id')->index('delivery_method_id');
            $table->float('rate', 10, 0);
        });

        Schema::create('item_size_labels', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id');
            $table->integer('size_id');
            $table->string('title');
            $table->string('strength');
            $table->tinyInteger('pack_size');
            $table->string('dosage');
            $table->text('warnings');
            $table->timestamps();
        });

        Schema::create('item_sizes__restrictions', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_size_id');
            $table->unsignedInteger('question_id')->index('item_sizes__restrictions_question_id_foreign');
            $table->string('restricting_answer')->nullable();
            $table->timestamps();

            $table->unique(['item_size_id', 'question_id'], 'item_sizes__restrictions__unique_item_size_id_question_id');
        });

        Schema::create('items', function (Blueprint $table) {
            $table->integer('id', true);
            $table->unsignedInteger('magento_entity_id')->nullable();
            $table->unsignedInteger('vs_parent_id')->nullable();
            $table->integer('form_id')->index('form_id');
            $table->boolean('disable_assessment_forms')->default(false);
            $table->string('epos_code')->index('epos_code');
            $table->string('name')->default('')->fulltext('name_2');
            $table->text('description');
            $table->text('full_description')->nullable();
            $table->string('search_name')->comment('stemmed words for searching');
            $table->text('search_description')->comment('stemmed words for searching');
            $table->double('price')->default(0)->index('price');
            $table->double('price_exvat');
            $table->double('sale_price')->index('sale_price');
            $table->double('sale_price_exvat');
            $table->double('sort_price')->default(0)->index('sort_price');
            $table->boolean('heavy')->default(false)->index('heavy');
            $table->boolean('hidden')->default(false)->index('hidden');
            $table->integer('stock')->default(0)->index('stock');
            $table->string('product_type', 50)->index('product_type');
            $table->boolean('id_check_required')->default(false);
            $table->integer('times_purchased')->index('times_purchased');
            $table->boolean('deleted')->index('deleted');
            $table->double('vat_rate');
            $table->string('weight');
            $table->boolean('vat_deductable')->index('vat_deductable');
            $table->string('google_category');
            $table->integer('num_sold')->index('num_sold');
            $table->integer('num_views')->default(0)->index('num_views');
            $table->integer('num_basket')->default(0)->index('num_basket');
            $table->unsignedInteger('custom_field_set_id')->index('custom_field_set_id');
            $table->dateTime('updated_at');
            $table->dateTime('created_at');
            $table->string('seo_title');
            $table->string('seo_keywords');
            $table->string('seo_description');
            $table->text('seo_schema_cache');
            $table->tinyInteger('use_category_faq')->default(1);
            $table->text('faq_cache');
            $table->integer('restriction_period_length')->nullable();
            $table->integer('restriction_per_period')->nullable();
            $table->integer('restriction_per_order')->nullable();
            $table->boolean('restriction_limit_once')->nullable();
            $table->boolean('subscription_is_allowed')->default(false);
            $table->boolean('subscription_can_be_paused')->default(true);
            $table->integer('subscription_max_number_of_issues')->nullable();
            $table->boolean('can_be_sold_separately')->default(true);
            $table->string('bundle_page_url')->nullable();
            $table->boolean('is_groupbuy_container')->default(false);
            $table->unsignedInteger('groupbuy_bundletype_id')->nullable()->index('items__groupbuy_bundletype_id');
            $table->boolean('basket_quantity_locked')->default(false);

            $table->index(['name'], 'name');
            $table->fullText(['name'], 'name_3');
            $table->fullText(['search_name', 'search_description'], 'search_name');
        });

        Schema::create('items__display_tags', function (Blueprint $table) {
            $table->increments('id');
            $table->char('group_name', 1)->default('a');
            $table->integer('item_id')->index('items__display_tags_item_id_foreign');
            $table->string('label');
            $table->unsignedInteger('icon_id')->nullable()->index('items__dt__icon_id');
            $table->timestamps();
        });

        Schema::create('items__out_of_stock_notifications', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->index('items__out_of_stock_notifications_item_id_foreign');
            $table->string('email', 100);
            $table->dateTime('notified_at')->nullable();
            $table->timestamps();
        });

        Schema::create('items_brands', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('item_id')->default(0)->index('item_id');
            $table->integer('brand_id')->default(0)->index('brand_id');
            $table->integer('order')->index('order');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('items_categories', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('item_id')->default(0)->index('item_id');
            $table->integer('cat_id')->default(0)->index('cat_id');
            $table->integer('order')->index('order');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('items_complimented', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->index();
            $table->integer('complimented_item_id')->index();
            $table->smallInteger('order');
            $table->dateTime('created_at');
            $table->dateTime('updated_at');
        });

        Schema::create('items_custom', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('item_id')->index('item_id');
            $table->string('slug')->index('old_slug');
            $table->string('cog_cost');
            $table->string('manufacturer');
            $table->string('vat_code', 5);
            $table->string('legal_code', 20);
            $table->string('gtin_13', 50);
            $table->text('contents');
            $table->text('indication');
            $table->string('dosage');
            $table->text('caution');
            $table->decimal('product_height', 5);
            $table->decimal('batch_number', 5);
            $table->decimal('product_depth', 5);
            $table->decimal('product_weight', 6, 3);
            $table->decimal('shipper_height', 5);
            $table->decimal('shipper_width', 5);
            $table->decimal('shipper_depth', 5);
            $table->decimal('shipper_weight', 6, 3);
            $table->string('needs_review')->index();
            $table->string('pharmacist_review')->index();
            $table->text('features');
            $table->text('faq');
            $table->text('side_effects');
            $table->text('information_leaflet');
            $table->text('reviews');
            $table->text('compare');
            $table->string('label_warnings')->index();
            $table->string('strength_field_name')->index();
            $table->string('schema_description')->index();
            $table->string('label_title')->index();
            $table->text('schema_alternate_name');
            $table->text('schema_non_proprietary_name');
            $table->text('schema_same_as');
            $table->text('schema_administration_route');
            $table->text('schema_dosage_form');
            $table->text('schema_active_ingredient');
            $table->text('schema_mechanism_of_action');
            $table->text('schema_related_drug');
            $table->text('schema_drug_class_name');
            $table->text('schema_drug_class_same_as');
            $table->text('schema_drug_unit');
            $table->text('schema_dose_schedule_unit');
            $table->text('schema_dose_schedule_value');
            $table->text('schema_dose_schedule_frequency');
            $table->text('schema_dose_target_population');
            $table->text('schema_indication_name');
            $table->text('schema_indication_same_as');
            $table->text('schema_indication_alternate_name');
            $table->text('schema_adverse_outcome');
            $table->text('schema_serious_adverse_outcome');
            $table->text('schema_contraindication');
            $table->text('schema_available_strengths');
            $table->text('schema_label_details');
            $table->text('schema_prescribing_info');
            $table->text('schema_brands');
            $table->text('schema_warnings');
            $table->text('schema_manufacturer_name');
            $table->text('schema_manufacturer_url');
            $table->string('product_custom_delivery')->index();
            $table->string('product_custom_packaging')->index();
            $table->string('allergens')->index();
            $table->string('ingredients')->index();
            $table->string('benefits')->index();
            $table->string('how_it_works')->index();
            $table->string('bbb')->index();
            $table->string('dispense_quantity')->index();
            $table->string('product_subtitle')->index();
            $table->string('description__product_card')->index();
            $table->string('barcode')->index();
            $table->string('is_recommended')->index();
            $table->string('medication_type')->index();
            $table->string('medication_effects')->index();
        });

        Schema::create('items_custom_multiple', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('item_id')->index('item_id');
            $table->string('field')->index('field');
            $table->string('value')->index('value');
        });

        Schema::create('items_faq', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->index();
            $table->integer('category_id');
            $table->string('question');
            $table->text('answer');
            $table->integer('order')->index();
            $table->timestamps();
        });

        Schema::create('items_images', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('item_id')->default(0)->index('item_id');
            $table->string('name')->default('');
            $table->text('old_location');
            $table->boolean('default')->default(false)->index('default');
            $table->boolean('hover')->default(false)->index('hover');
            $table->boolean('zoom')->default(false)->index('zoom');
            $table->boolean('small')->default(false);
            $table->string('type')->index('type');
            $table->string('colour', 10);
            $table->integer('order')->index('order');
            $table->dateTime('updated_at');
            $table->dateTime('created_at');
            $table->string('alt')->nullable()->index();
        });

        Schema::create('items_incompatible', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedBigInteger('main_product')->index();
            $table->unsignedBigInteger('linked_product')->index();
            $table->string('level')->nullable();
            $table->timestamps();
            $table->string('message')->nullable();
        });

        Schema::create('items_info', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->index('items_info_item_id_foreign');
            $table->string('title')->nullable();
            $table->boolean('is_content_itemised')->default(false);
            $table->text('content')->nullable();
            $table->integer('order')->default(0);
            $table->timestamps();
        });

        Schema::create('items_ingredients', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id');
            $table->integer('size_id');
            $table->integer('ingredient_id');

            $table->index(['item_id', 'size_id']);
        });

        Schema::create('items_keywords', function (Blueprint $table) {
            $table->integer('item_id')->index('fk_items_keywords_items1_idx');
            $table->unsignedInteger('keyword_id')->index('fk_items_keywords_keywords1_idx');
            $table->string('keyword')->nullable();
            $table->string('search_keyword')->nullable()->fulltext('search_keyword');

            $table->primary(['item_id', 'keyword_id']);
        });

        Schema::create('items_properties', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('property_id')->index('property_id');
            $table->unsignedInteger('property_type_id')->index('property_type_id');
            $table->timestamp('timestamp')->useCurrentOnUpdate()->useCurrent();
            $table->unsignedInteger('item_id')->index('item_id');
        });

        Schema::create('items_seo_schema', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->index();
            $table->string('item_type')->index();
            $table->integer('parent_id')->nullable()->index();
            $table->integer('lft')->index();
            $table->integer('rgt')->index();
            $table->integer('depth');
            $table->string('type');
            $table->string('field')->index();
            $table->string('value')->nullable();
            $table->timestamps();
        });

        Schema::create('items_sizes', function (Blueprint $table) {
            $table->integer('id', true);
            $table->unsignedInteger('magento_entity_id')->nullable();
            $table->unsignedInteger('vs_child_id')->nullable();
            $table->integer('item_id')->default(0)->index('item_id');
            $table->string('size')->default('')->index('size');
            $table->string('quantity');
            $table->integer('pack_size')->default(1);
            $table->integer('label_count')->default(0);
            $table->string('strength');
            $table->double('price')->index('price');
            $table->double('price_exvat');
            $table->double('sale_price')->index('sale_price');
            $table->double('sale_price_exvat');
            $table->integer('stock')->default(0)->index('stock');
            $table->integer('order')->default(0);
            $table->boolean('manual')->index('manual');
            $table->string('PLU', 50)->index('PLU');
            $table->string('ROOT_PLU', 50)->index('ROOT_PLU');
            $table->string('ean')->index('ean');
            $table->boolean('hide')->index('hide');
            $table->binary('epos_info')->nullable();
            $table->dateTime('created_at');
            $table->dateTime('updated_at');
            $table->integer('form_id')->nullable();
        });

        Schema::create('items_sizes_custom', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('item_id')->index('item_id');
            $table->integer('size_id')->index('size_id');
            $table->decimal('cog_cost', 5);
            $table->string('manufacturer');
            $table->string('vat_code', 5);
            $table->string('legal_code', 20);
            $table->string('gtin_13', 50);
            $table->text('contents');
            $table->text('indication');
            $table->text('dosage');
            $table->text('caution');
            $table->decimal('product_height', 5);
            $table->decimal('batch_number', 5);
            $table->decimal('product_depth', 5);
            $table->decimal('product_weight', 5);
            $table->decimal('shipper_height', 5);
            $table->decimal('shipper_width', 5);
            $table->decimal('shipper_depth', 5);
            $table->decimal('shipper_weight', 5);
            $table->string('needs_review')->index();
            $table->string('pharmacist_review')->index();
            $table->text('features');
            $table->string('faq')->index();
            $table->string('side_effects')->index();
            $table->string('information_leaflet')->index();
            $table->string('reviews')->index();
            $table->string('compare')->index();
            $table->string('label_warnings')->index();
            $table->string('strength_field_name')->index();
            $table->string('schema_description')->index();
            $table->string('label_title')->index();
            $table->text('schema_alternate_name');
            $table->text('schema_non_proprietary_name');
            $table->text('schema_same_as');
            $table->text('schema_administration_route');
            $table->text('schema_dosage_form');
            $table->text('schema_active_ingredient');
            $table->text('schema_mechanism_of_action');
            $table->text('schema_related_drug');
            $table->text('schema_drug_class_name');
            $table->text('schema_drug_class_same_as');
            $table->text('schema_drug_unit');
            $table->text('schema_dose_schedule_unit');
            $table->text('schema_dose_schedule_value');
            $table->text('schema_dose_schedule_frequency');
            $table->text('schema_dose_target_population');
            $table->text('schema_indication_name');
            $table->text('schema_indication_same_as');
            $table->text('schema_indication_alternate_name');
            $table->text('schema_adverse_outcome');
            $table->text('schema_serious_adverse_outcome');
            $table->text('schema_contraindication');
            $table->text('schema_available_strengths');
            $table->text('schema_label_details');
            $table->text('schema_prescribing_info');
            $table->text('schema_brands');
            $table->text('schema_warnings');
            $table->text('schema_manufacturer_name');
            $table->text('schema_manufacturer_url');
            $table->string('alt_tag')->index();
            $table->string('product_custom_delivery')->index();
            $table->string('product_custom_packaging')->index();
            $table->string('allergens')->index();
            $table->string('ingredients')->index();
            $table->string('benefits')->index();
            $table->string('how_it_works')->index();
            $table->string('bbb')->index();
            $table->string('dispense_quantity')->index();
            $table->string('product_subtitle')->index();
            $table->string('description__product_card')->index();
            $table->string('barcode')->index();
            $table->string('is_recommended')->index();
            $table->string('medication_type')->index();
            $table->string('medication_effects')->index();
        });

        Schema::create('judopay_payments', function (Blueprint $table) {
            $table->increments('id');
            $table->string('order_id')->index();
            $table->string('receipt_id', 50);
            $table->string('type', 50);
            $table->string('acquirer_transaction_id', 50)->nullable();
            $table->double('amount', 8, 2);
            $table->string('currency', 5);
            $table->string('trx_ref', 50);
            $table->string('method')->nullable();
            $table->timestamps();
        });

        Schema::create('keywords', function (Blueprint $table) {
            $table->increments('id');
            $table->string('keyword')->unique('keyword');
        });

        Schema::create('languages', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('name', 63);
            $table->boolean('active')->default(true)->index('active');
            $table->string('code', 12)->index('code');
            $table->string('localisation', 12)->index('localisation');
            $table->string('text_direction', 10)->index('text_direction');
            $table->text('image');
            $table->string('custom_code', 63);
            $table->boolean('default')->index('default')->comment('For reference.');
            $table->string('country', 63);
            $table->string('time_format', 63);
            $table->string('date_format', 63);
            $table->text('lang_file');
            $table->string('domain', 64);
            $table->integer('order')->default(0)->index('order');
            $table->boolean('public')->default(true)->index('public');
        });

        Schema::create('listing_cache', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('setup_hash', 32)->index('setup_hash');
            $table->longText('result');
            $table->text('query');
            $table->text('params');
            $table->timestamp('created')->useCurrent()->index('timestamp');
            $table->timestamp('updated')->default('0000-00-00 00:00:00');
            $table->timestamp('last_fetched')->default('0000-00-00 00:00:00')->index('last_fetched');
            $table->integer('num_of_fetches')->default(0);
            $table->integer('num_of_total_fetches')->default(0);
            $table->double('refresh_time_in_seconds');
        });

        Schema::create('log_id_checks', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')->index();
            $table->integer('member_id')->index();
            $table->tinyInteger('passed')->index();
            $table->text('request');
            $table->text('response');
            $table->text('raw_response')->nullable();
            $table->string('endpoint');
            $table->timestamps();
        });

        Schema::create('medical_conditions', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->timestamps();
        });

        Schema::create('medications', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->index('name');
            $table->timestamps();
        });

        Schema::create('member_oauth_providers', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('member_id')->index();
            $table->string('provider');
            $table->string('key');
            $table->timestamps();

            $table->index(['id']);
        });

        Schema::create('members', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('referrer_member_id')->nullable()->index('members_referrer_member_id_foreign');
            $table->unsignedInteger('magento_entity_id')->nullable();
            $table->unsignedBigInteger('unio_internal_id')->nullable();
            $table->boolean('is_external')->default(false);
            $table->string('email')->nullable()->unique();
            $table->string('password')->index();
            $table->boolean('is_activated')->default(true);
            $table->string('contact_no')->nullable();
            $table->string('hash');
            $table->text('conditions');
            $table->text('medications');
            $table->text('allergies');
            $table->integer('basket_id');
            $table->boolean('optout');
            $table->tinyInteger('prescription_reminder_email');
            $table->tinyInteger('prescription_reminder_sms');
            $table->dateTime('last_login');
            $table->dateTime('last_failed_login');
            $table->smallInteger('failed_login_count');
            $table->timestamps();
            $table->dateTime('first_login')->nullable();
            $table->dateTime('id_verified')->nullable();
            $table->string('verification_code', 10);
            $table->dateTime('verification_code_expiry');
            $table->string('phone_prefix');
            $table->dateTime('passed_id_check')->nullable()->index();
            $table->smallInteger('dob_date')->nullable();
            $table->smallInteger('dob_month')->nullable();
            $table->smallInteger('dob_year')->nullable();
            $table->string('gender', 20)->index();
            $table->text('doctor_surgery');
            $table->boolean('has_nominated_client_as_pharmacy_of_choice')->default(false);
            $table->timestamp('confirmed_terms_at')->nullable();

            $table->index(['id']);
            $table->index(['email']);
        });

        Schema::create('members_addresses', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('member_id')->index();
            $table->enum('type', ['billing', 'shipping']);
            $table->string('title', 10);
            $table->string('firstname');
            $table->string('middle_name')->nullable();
            $table->string('lastname');
            $table->string('address1');
            $table->string('address2');
            $table->string('city');
            $table->string('state');
            $table->string('country');
            $table->string('postcode');
            $table->timestamps();

            $table->index(['id']);
        });

        Schema::create('members_info', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedInteger('member_id')->unique();
            $table->unsignedInteger('multisite__site_id')->nullable()->index('members_info_multisite__site_id_foreign');
            $table->timestamps();
        });

        Schema::create('members_items_preferences', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('member_id')->index();
            $table->integer('cat_id');
            $table->integer('brand_id');
            $table->integer('item_id');
            $table->timestamps();

            $table->index(['id']);
        });

        Schema::create('messenger_messages', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('member_id');
            $table->integer('thread_id');
            $table->text('text');
            $table->tinyInteger('satisfactory')->default(0);
            $table->text('comment');
            $table->timestamps();
            $table->softDeletes();
            $table->dateTime('read_at')->nullable();
            $table->boolean('admin');
            $table->tinyInteger('private')->default(0);
        });

        Schema::create('messenger_messages_read', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('member_id');
            $table->integer('message_id');
            $table->integer('thread_id');
            $table->dateTime('read_at');
            $table->timestamps();
        });

        Schema::create('messenger_participants', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('member_id');
            $table->integer('thread_id');
            $table->timestamps();
            $table->tinyInteger('admin')->index();
        });

        Schema::create('messenger_templates', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->text('text');
            $table->timestamps();
            $table->integer('role')->index();
            $table->string('category')->nullable()->index();
        });

        Schema::create('messenger_threads', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')->default(0)->index();
            $table->string('url');
            $table->timestamps();
            $table->softDeletes();
            $table->tinyInteger('assigned')->index();
            $table->unsignedInteger('assigned_participant_id')->nullable()->index('messenger_threads_assigned_participant_id_foreign');
            $table->integer('category');
        });

        Schema::create('multisite__base_entities', function (Blueprint $table) {
            $table->increments('id');
            $table->enum('concrete_entity_type', ['shop_item', 'cms_page', 'shop_category', 'shop_brand', 'site_menu']);
            $table->integer('concrete_entity_id');
            $table->enum('mode', ['single-site', 'multi-site']);
            $table->timestamps();

            $table->unique(['concrete_entity_id', 'concrete_entity_type'], 'multisite__default_entities__unique_entities');
        });

        Schema::create('multisite__custom_entities', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('parent_entity_id')->index('multisite__custom_entities_parent_entity_id_foreign');
            $table->integer('concrete_entity_id');
            $table->unsignedInteger('site_id');
            $table->timestamps();

            $table->unique(['site_id', 'parent_entity_id'], 'multisite__custom_entities__unique_site_entities');
        });

        Schema::create('multisite__sites', function (Blueprint $table) {
            $table->unsignedInteger('id')->primary();
            $table->string('name', 64)->unique();
        });

        Schema::create('nhs_dosages', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->decimal('quantity_per_day', 5);
            $table->unsignedInteger('member_id')->nullable()->index();
            $table->timestamps();

            $table->unique(['name', 'member_id']);
        });

        Schema::create('nhs_medical_categories', function (Blueprint $table) {
            $table->increments('id');
            $table->string('label');
        });

        Schema::create('nhs_medications', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->unique();
            $table->timestamps();
        });

        Schema::create('nhs_member_doctor_surgeries', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('nhs_member_id')->index('nhs_member_doctor_surgeries_nhs_member_id_foreign');
            $table->string('practice_name');
            $table->string('address_1');
            $table->string('address_2')->nullable();
            $table->string('city');
            $table->string('county')->nullable();
            $table->string('postcode')->nullable();
            $table->string('phone')->nullable();
            $table->string('email')->nullable();
            $table->timestamps();
        });

        Schema::create('nhs_member_prescriptions', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('nhs_member_id')->index();
            $table->string('name');
            $table->string('quantity');
            $table->string('quantity_per_dose');
            $table->unsignedInteger('dosage_id')->nullable()->index('nhs_member_prescriptions_dosage_id_foreign');
            $table->date('refill_date')->nullable();
            $table->date('refill_date_new')->nullable();
            $table->tinyInteger('one_off_prescription')->default(0);
            $table->text('additional_info')->nullable();
            $table->timestamps();

            $table->unique(['name', 'nhs_member_id'], 'nhs_member_prescriptions_name_unique');
        });

        Schema::create('nhs_members', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('member_id')->unique();
            $table->unsignedInteger('doctor_surgery_id')->nullable()->index('doctor_surgery_id');
            $table->string('job_title')->nullable();
            $table->string('firstname');
            $table->string('lastname');
            $table->date('dob')->nullable();
            $table->string('nhs_number');
            $table->string('gender', 20);
            $table->date('reminder_date');
            $table->tinyInteger('is_pay_for_prescriptions');
            $table->text('pay_explanation');
            $table->date('exemption_expiry_date')->nullable();
            $table->string('exemption_proof_file', 64)->nullable();
            $table->tinyInteger('is_property_safe');
            $table->tinyInteger('is_patient');
            $table->boolean('is_vulnerable')->default(false);
            $table->boolean('is_hcp')->default(false);
            $table->boolean('is_professional_carer')->default(false);
            $table->timestamps();
        });

        Schema::create('nhs_members_nhs_medical_categories', function (Blueprint $table) {
            $table->unsignedInteger('nhs_member_id');
            $table->unsignedInteger('medical_category_id')->index('nhs_members_nhs_medical_categories_medical_category_id_foreign');

            $table->unique(['nhs_member_id', 'medical_category_id'], 'unique_nhs_member_medical_categories');
        });

        Schema::create('nhs_prescription_items', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('prescription_id')->index('nhs_prescription_items_prescription_id_foreign');
            $table->string('name');
            $table->unsignedBigInteger('nhs_unilexicon_product_id')->nullable();
            $table->integer('quantity')->nullable();
            $table->date('refill_date')->nullable();
            $table->boolean('one_off_prescription')->default(false);
            $table->unsignedInteger('frequency_days')->nullable();
            $table->text('additional_info')->nullable();
            $table->timestamps();
        });

        Schema::create('nhs_prescriptions', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('copy_of_prescription_id')->nullable()->index('nhs_prescriptions_copy_of_prescription_id_foreign');
            $table->unsignedBigInteger('unio_internal_id')->nullable();
            $table->boolean('is_external')->default(false);
            $table->char('uuid', 36)->unique();
            $table->unsignedInteger('nhs_member_id')->nullable()->index('nhs_prescriptions_nhs_member_id_foreign');
            $table->enum('status', ['draft', 'pending', 'complete'])->default('draft');
            $table->timestamps();
        });

        Schema::create('nhs_reminders', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')->index();
            $table->integer('order_item_id')->index();
            $table->integer('member_id')->index();
            $table->text('medication_name');
            $table->boolean('send_email');
            $table->boolean('send_sms');
            $table->dateTime('refill_date')->index('nhs_reminders_reminder_date_index');
            $table->dateTime('sent_at')->nullable()->index();
            $table->timestamps();
        });

        Schema::create('nhs_unilexicon_products', function (Blueprint $table) {
            $table->unsignedBigInteger('id')->primary();
            $table->string('invalid');
            $table->string('gtin', 32);
            $table->string('tariffs', 32);
            $table->string('price', 32);
            $table->text('name');
            $table->string('short', 64);
            $table->string('supplier');
            $table->string('mtc__display_name')->index();
            $table->unsignedTinyInteger('number_of_charges')->default(1);
        });

        Schema::create('order', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('parent_id')->nullable()->index('order_parent_id_foreign');
            $table->unsignedInteger('magento_entity_id')->nullable();
            $table->integer('paid')->index('paid');
            $table->string('worldpay_order_code')->nullable();
            $table->string('worldpay_session_id')->nullable();
            $table->integer('basket_id')->default(0)->index('basket_id');
            $table->unsignedInteger('subscription_id')->nullable()->index('order__subscription_id');
            $table->dateTime('date')->useCurrent();
            $table->string('delivery_name')->default('');
            $table->integer('delivery_method_id')->index();
            $table->double('delivery_cost')->default(0);
            $table->boolean('coupon')->default(false);
            $table->double('cost_total')->default(0)->index('cost_total');
            $table->double('cost_total_exvat');
            $table->unsignedInteger('status')->default(0)->index('status');
            $table->integer('member')->default(0)->index('member');
            $table->integer('admin_id')->index();
            $table->string('coupon_code', 50)->index('coupon_code');
            $table->string('type', 50)->index('type');
            $table->string('ref');
            $table->text('keywords');
            $table->boolean('failed')->index('failed');
            $table->text('notes');
            $table->string('order_ref');
            $table->timestamp('timestamp')->useCurrent();
            $table->string('ga_ga', 555)->comment('google cookie data - used for analytics tracking');
            $table->dateTime('exported');
            $table->string('checkout', 6);
            $table->tinyInteger('blacklisted')->index();
            $table->dateTime('trustpilot_sent');
            $table->tinyInteger('flag')->comment('1 - brown list, 2 - black list');
            $table->integer('message_id');
            $table->string('tracking_no');
            $table->string('shipping_label');
            $table->dateTime('rm_exported');
            $table->dateTime('failed_id_check')->nullable()->index();
            $table->tinyInteger('id_check_confirmed')->nullable()->index();
            $table->dateTime('doctor_approved')->nullable();
            $table->dateTime('pharmacist_approved')->nullable();
            $table->dateTime('doctor_rejected')->nullable();
            $table->dateTime('pharmacist_rejected')->nullable();
            $table->text('doctor_note');
            $table->text('pharmacist_note');
            $table->text('patient_notes');
            $table->text('rm_error')->nullable();
            $table->boolean('rm_failed')->default(false);
            $table->boolean('archived')->default(false)->index();
            $table->text('delivery_instructions')->nullable();
        });

        Schema::create('order_address', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->default(0)->index('order_id');
            $table->set('type', ['billing', 'shipping'])->default('')->index('type');
            $table->string('gender', 10);
            $table->string('title', 10);
            $table->string('firstname')->default('');
            $table->string('middle_name')->nullable();
            $table->string('lastname')->default('');
            $table->string('address1')->default('');
            $table->string('address2')->default('');
            $table->string('city')->default('');
            $table->string('state');
            $table->string('country')->default('');
            $table->string('postcode')->default('');
            $table->timestamp('timestamp')->useCurrent();
            $table->string('notes')->nullable();
        });

        Schema::create('order_archive_items', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_item_id')->index();
            $table->integer('order_id')->index();
            $table->integer('item_id')->index();
            $table->integer('assessment_id')->index();
            $table->string('item_name');
            $table->double('item_price');
            $table->double('item_price_exvat');
            $table->integer('quantity');
            $table->string('size');
            $table->string('PLU', 50);
            $table->boolean('tosend');
            $table->dateTime('reminder_date')->nullable();
            $table->dateTime('sent');
            $table->string('refund_amount', 7);
            $table->double('discount');
            $table->integer('sizeid');
            $table->boolean('nhs_prescription');
            $table->dateTime('timestamp');
            $table->double('sale_item');
            $table->integer('vat_deductable');
            $table->double('price_paid');
            $table->double('price_paid_exvat');
            $table->tinyInteger('approved');
            $table->timestamps();
            $table->string('pharmacy_label_text')->nullable();
        });


        Schema::create('order_contact', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->index('order_id');
            $table->string('subject');
            $table->text('message');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('order_coupon', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->default(0)->index('order_id');
            $table->string('code')->default('')->index('code');
            $table->boolean('freedelivery')->default(false)->index('freedelivery');
            $table->double('amountoff')->default(0);
            $table->double('percentoff')->default(0);
            $table->double('deduct')->default(0);
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('order_discounts', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id');
            $table->decimal('discount_amount', 10);
            $table->string('discount_type');
            $table->integer('discount_id');
            $table->string('discount_name');
            $table->string('payment_type');
        });

        Schema::create('order_evopay', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id');
            $table->decimal('amount_paid', 10);
            $table->string('tnx_id', 50);
            $table->dateTime('paid_time');
            $table->timestamps();

            $table->index(['order_id', 'tnx_id']);
        });

        Schema::create('order_info', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->default(0)->index('order_id');
            $table->string('email')->nullable()->default('');
            $table->string('contact_no')->nullable()->default('');
            $table->text('message');
            $table->text('member_conditions');
            $table->text('member_medications');
            $table->text('member_allergies');
            $table->integer('newsletter');
            $table->string('how_did_you_hear_about_us')->default('');
            $table->string('how_did_you_hear_about_us_text')->default('');
            $table->timestamp('timestamp')->useCurrent();
            $table->string('phone_prefix');
            $table->smallInteger('dob_date')->nullable();
            $table->smallInteger('dob_month')->nullable();
            $table->smallInteger('dob_year')->nullable();
            $table->longText('assessment_ids')->nullable();
            $table->unsignedInteger('multisite__site_id')->nullable()->index('order_info_multisite__site_id_foreign');
        });

        Schema::create('order_items', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->default(0)->index('order_id');
            $table->integer('item_id')->default(0)->index('item_id');
            $table->unsignedInteger('bundle_id')->nullable()->index('order_items__bundle_id');
            $table->unsignedInteger('prescription_item_id')->nullable()->index('order_items_prescription_item_id_foreign');
            $table->boolean('has_requested_subscription')->default(false);
            $table->integer('assessment_id');
            $table->string('item_name')->default('');
            $table->double('item_price')->default(0);
            $table->double('item_price_exvat');
            $table->integer('quantity')->default(0);
            $table->integer('recommended_quantity')->default(0);
            $table->string('size')->default('');
            $table->string('PLU', 50);
            $table->boolean('tosend');
            $table->dateTime('reminder_date')->nullable();
            $table->dateTime('sent');
            $table->string('refund_amount', 7);
            $table->double('discount');
            $table->integer('sizeid');
            $table->boolean('nhs_prescription')->default(false)->index();
            $table->timestamp('timestamp')->useCurrent();
            $table->double('sale_item');
            $table->integer('vat_deductable');
            $table->double('price_paid');
            $table->double('price_paid_exvat');
            $table->tinyInteger('approved');
            $table->string('pharmacy_label_text')->nullable();
        });

        Schema::create('order_items_meta', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id');
            $table->integer('item_id');
            $table->integer('order_item_id');
            $table->string('meta_key');
            $table->longText('meta_value');
            $table->timestamps();

            $table->index(['order_id', 'item_id', 'order_item_id']);
        });

        Schema::create('order_note', function (Blueprint $table) {
            $table->integer('id', true);
            $table->unsignedInteger('magento_entity_id')->nullable();
            $table->integer('order_id')->index('order_id');
            $table->text('note');
            $table->timestamp('timestamp')->useCurrentOnUpdate()->useCurrent();
        });

        Schema::create('order_paypal', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->default(0)->index('order_id');
            $table->string('mc_gross')->default('');
            $table->string('txn_id')->default('');
            $table->string('payer_email')->default('');
            $table->integer('item_number')->default(0);
            $table->string('payment_type')->default('');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('order_protx', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->default(0)->index('order_id');
            $table->string('VendorTxCode')->default('');
            $table->string('VPSTxId')->default('');
            $table->string('TxAuthNo')->default('');
            $table->double('Amount')->default(0);
            $table->string('AVSCV2')->default('');
            $table->string('AddressResult')->default('');
            $table->string('PostCodeResult')->default('');
            $table->string('CV2Result')->default('');
            $table->string('GiftAid')->default('');
            $table->string('3DSecureStatus')->default('');
            $table->string('CAVV')->default('');
            $table->string('SecurityKey', 50);
            $table->integer('released');
            $table->double('released_amount');
            $table->double('AmountRemaining');
            $table->timestamp('timestamp')->useCurrent();
            $table->string('type')->nullable();
            $table->dateTime('captured_at')->nullable();
        });

        Schema::create('order_realex', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')->index('order_realex_order_id_foreign');
            $table->decimal('amount_paid', 10);
            $table->string('tnx_id', 50);
            $table->dateTime('paid_time')->nullable();
            $table->timestamps();
        });

        Schema::create('order_refund_items', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('refund_id')->index();
            $table->integer('order_id')->index();
            $table->integer('order_item_id')->index();
            $table->integer('item_id')->index();
            $table->integer('quantity');
            $table->decimal('amount_refunded');
            $table->timestamps();
        });

        Schema::create('order_refunds', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')->index();
            $table->decimal('delivery_refund_amount');
            $table->string('reference');
            $table->string('note');
            $table->timestamps();
        });

        Schema::create('order_shipment', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->index('order_id');
            $table->date('date');
            $table->string('carrier');
            $table->string('service')->index('service');
            $table->string('tracking_no');
            $table->timestamp('timestamp')->useCurrent();
            $table->integer('order_status_id');
            $table->string('label_format')->nullable();
            $table->mediumText('label')->nullable();
            $table->text('shipment_data')->nullable();
        });

        Schema::create('order_shipment_items', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_shipment_id')->index('order_shipment_id');
            $table->integer('order_item_id')->index('order_item_id');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('order_status', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->index('order_id');
            $table->integer('status')->index('status');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('order_status_email_text', function (Blueprint $table) {
            $table->unsignedInteger('status_id')->primary();
            $table->text('text');
        });

        Schema::create('order_statuses', function (Blueprint $table) {
            $table->unsignedInteger('id')->primary();
            $table->string('name', 64);
            $table->string('mapping_code', 32)->nullable();
        });

        Schema::create('order_stripe', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->index('order_id');
            $table->string('charge_id')->index('charge_id');
            $table->string('order_ref')->index('order_ref');
            $table->timestamp('timestamp')->useCurrentOnUpdate()->useCurrent()->index('timestamp');
            $table->double('amount')->index('amount');
        });

        Schema::create('order_worldpay', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order_id')->index('order_id');
            $table->string('order_ref', 128)->index('order_ref');
            $table->string('transId', 555)->index('transId');
            $table->timestamp('timestamp')->useCurrent()->index('timestamp');
            $table->double('amount')->index('amount');
        });

        Schema::create('page_cache', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('page_id')->index('page_id');
            $table->longText('object');
            $table->timestamp('timestamp')->useCurrentOnUpdate()->useCurrent();
            $table->string('url', 2024)->index('url');
            $table->string('language', 5)->index('language');
        });

        Schema::create('page_list', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('page_id')->index('page_id');
            $table->string('name', 64);
            $table->string('type', 32);
            $table->integer('setup')->default(0);
            $table->boolean('lock')->default(false);
            $table->text('info');
            $table->integer('order')->default(0)->index('order');
            $table->boolean('atleast_one')->default(true);
            $table->boolean('allow_more')->default(false);
            $table->boolean('add_list_item_to_top')->default(false);
            $table->string('more_text', 32)->default('Add another one');
            $table->text('settings');
            $table->integer('allow_more_max')->default(200);
            $table->integer('allow_more_min')->default(1);
            $table->text('data1');
            $table->text('data2');
            $table->text('data3');
        });

        Schema::create('page_list_item', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('page_id')->index('page_id');
            $table->integer('page_list_id')->index('page_list_id');
            $table->string('type', 32)->index('type');
            $table->integer('order')->index('order');
            $table->string('data1', 128)->index('data1');
        });

        Schema::create('page_list_item_data', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('page_id')->index('page_id');
            $table->integer('page_list_id')->index('page_list_id');
            $table->integer('page_list_item_id')->index('page_list_item_id');
            $table->string('name', 64)->index('name');
            $table->text('info');
            $table->integer('order')->default(0)->index('order');
            $table->string('type', 32)->default('text')->index('type');
            $table->string('datatype', 12);
            $table->integer('setup')->default(0);
            $table->boolean('lock')->default(false);
            $table->longText('value');
            $table->dateTime('datetime_value')->index('datetime_value');
            $table->integer('int_value')->index('int_value');
            $table->boolean('tinyint_value')->index('tinyint_value');
            $table->float('float_value', 10, 0)->index('float_value');
            $table->string('varchar_value', 1024)->index('varchar_value');
            $table->text('regex');
            $table->string('mime');
            $table->tinyInteger('required')->default(0);
            $table->text('settings');
            $table->string('css_class', 125);
            $table->text('options');
            $table->string('language', 5)->index('language');
            $table->char('uuid', 16)->index('uuid');
            $table->longText('data1');
            $table->longText('data2');
            $table->string('data3', 5)->index('data3');
        });

        Schema::create('page_options', function (Blueprint $table) {
            $table->integer('id', true)->index('id');
            $table->string('key')->unique();
            $table->string('name')->nullable();
            $table->mediumInteger('order')->nullable();
            $table->boolean('is_admin')->default(false);
        });

        Schema::create('pages', function (Blueprint $table) {
            $table->integer('id', true);
            $table->char('uuid', 16)->index('uuid');
            $table->integer('sub_id')->default(0)->index('sub_id');
            $table->integer('order')->index('order');
            $table->string('type', 32)->default('default')->index('type');
            $table->string('title', 2048)->index('title');
            $table->text('slug');
            $table->text('url');
            $table->boolean('innav')->default(true)->index('innav');
            $table->boolean('lock')->default(false);
            $table->boolean('hide_in_cms')->default(false);
            $table->boolean('allow_delete')->default(false);
            $table->boolean('allow_update')->default(true);
            $table->boolean('redchild')->default(false);
            $table->boolean('listing_container')->default(false)->index('listing_container');
            $table->integer('draft_for_page_id')->default(0)->index('draft_for_page_id');
            $table->boolean('searchable')->default(true)->index('searchable');
            $table->text('search_title');
            $table->text('search_content');
            $table->longText('content');
            $table->dateTime('created')->index('created');
            $table->timestamp('updated')->useCurrent()->index('updated');
            $table->boolean('published')->default(false)->index('published');
            $table->string('template', 64)->default('default');
            $table->string('layout', 32)->default('');
            $table->longText('logic');
            $table->text('seo_title');
            $table->text('seo_keywords');
            $table->text('seo_description');
            $table->boolean('frontpage')->default(false)->index('frontpage');
            $table->boolean('contact_page')->default(false);
            $table->text('children_order');
            $table->boolean('directly_viewable')->default(true)->index('directly_viewable');
            $table->boolean('history_lock')->default(false)->index('history_lock');
            $table->boolean('review_status')->default(false)->index('review_status');
            $table->boolean('noindex')->default(false)->index('noindex');
            $table->boolean('ajax')->default(false)->index('ajax');
            $table->boolean('section')->default(false)->index('section');
            $table->text('versions');
            $table->string('author_name', 32);
            $table->integer('author_admin_user_id')->index('author_admin_user_id');
            $table->string('comment');
            $table->text('data1');
            $table->text('data2');
            $table->text('data3');

            $table->fullText(['search_title', 'search_content'], 'search_title');
        });

        Schema::create('placeholder_images', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('name');
            $table->string('value');
            $table->dateTime('updated_at');
            $table->dateTime('created_at');

            $table->unique(['id'], 'id');
        });

        Schema::create('private_prescription_dosages', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->unique('private_prescriptions_dosages_name_unique');
            $table->decimal('quantity_per_day', 5);
            $table->timestamps();
        });

        Schema::create('private_prescription_medications', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->string('pack_size', 50)->nullable();
            $table->decimal('price');
            $table->decimal('store_cost')->nullable();
            $table->timestamps();
        });

        Schema::create('properties', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('property_type_id')->index('property_type_id');
            $table->string('name')->index('name');
            $table->integer('order')->index('order');
            $table->timestamp('timestamp')->useCurrentOnUpdate()->useCurrent();
        });

        Schema::create('property_types', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->index('name');
            $table->timestamp('timestamp')->useCurrentOnUpdate()->useCurrent();
            $table->integer('order')->default(0)->index('order');
        });

        Schema::create('realex_responses', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')->index('realex_responses_order_id_foreign');
            $table->string('type');
            $table->text('response');
            $table->timestamps();
        });

        Schema::create('sagepay_responses', function (Blueprint $table) {
            $table->dateTime('date');
            $table->string('VendorTxCode', 50);
            $table->text('request');
            $table->string('StatusDetail');
            $table->text('response');
        });

        Schema::create('seo_canonicals', function (Blueprint $table) {
            $table->increments('id');
            $table->string('path', 1024)->index('path');
            $table->text('canonical_path');
        });

        Schema::create('seo_defaults', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->index('name');
            $table->text('path');
            $table->text('title');
            $table->text('description');
        });

        Schema::create('seo_headings', function (Blueprint $table) {
            $table->increments('id');
            $table->string('path', 1024)->index('path');
            $table->string('text');
        });

        Schema::create('seo_pages', function (Blueprint $table) {
            $table->increments('id');
            $table->string('path', 1024)->index('path');
            $table->text('title');
            $table->text('description');
        });

        Schema::create('seo_redirects', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('old_url')->index('old_url');
            $table->string('new_url');
            $table->integer('hits')->index('hits');
            $table->string('code')->index('code');
        });

        Schema::create('seo_schema_templates', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->index();
            $table->timestamps();
        });

        Schema::create('seo_settings', function (Blueprint $table) {
            $table->string('key', 50)->primary();
            $table->string('value', 50);
            $table->text('description');
        });

        Schema::create('settings', function (Blueprint $table) {
            $table->increments('id');
            $table->string('key', 100)->default('');
            $table->string('environment', 50)->default('');
            $table->enum('type', ['', 'bool', 'int', 'float', 'string', 'array', 'object', 'null']);
            $table->text('value')->nullable();
            $table->string('module', 25)->nullable()->index('module');
            $table->text('description')->nullable();
            $table->timestamp('created_at')->useCurrentOnUpdate()->useCurrent();
            $table->timestamp('updated_at')->useCurrentOnUpdate()->nullable();

            $table->unique(['key', 'environment'], 'key_environment');
        });

        Schema::create('share_buttons', function (Blueprint $table) {
            $table->increments('id');
            $table->string('social_network');
            $table->string('social_button_link');
            $table->string('social_button_class');
            $table->string('social_button_text');
            $table->string('social_button_icon');
            $table->integer('order')->default(0);
            $table->integer('hide');
            $table->timestamps();
        });

        Schema::create('slides', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('slidelink');
            $table->string('title');
            $table->string('slideimage');
            $table->integer('order');
            $table->text('description');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('standard_template_letters', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('title');
            $table->text('template');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('static_blocks', function (Blueprint $table) {
            $table->increments('id');
            $table->string('type');
            $table->string('name');
            $table->integer('published');
        });

        Schema::create('static_blocks_instances', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('static_block_id');
            $table->string('url');
            $table->string('alt_text');
            $table->integer('size');
            $table->string('image');
            $table->text('content');
            $table->timestamps();
        });

        Schema::create('static_blocks_sizes', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->integer('width');
            $table->integer('height');
            $table->integer('crop');
            $table->integer('forced');
        });

        Schema::create('subscription_items', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('subscription_id')->index('subscription_items_subscription_id_foreign');
            $table->integer('shop_item_id')->index('subscription_items_shop_item_id_foreign');
            $table->integer('shop_item_size_id')->nullable()->index('subscription_items__shop_item_size_id');
            $table->timestamps();
        });

        Schema::create('subscription_preferences', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('item_id')->nullable();
            $table->integer('item_size_id')->nullable()->index('subscription_options_item_size_id_foreign');
            $table->boolean('is_allowed')->default(false);
            $table->boolean('is_required')->default(false);
            $table->boolean('can_be_paused')->default(true);
            $table->integer('max_number_of_issues')->nullable()->default(99);
            $table->integer('interval_multiplier')->default(1);
            $table->enum('interval', ['days', 'weeks', 'months'])->default('months');
            $table->timestamps();

            $table->unique(['item_id', 'item_size_id'], 'subscription_options_item_id_item_size_id_unique');
        });

        Schema::create('subscriptions', function (Blueprint $table) {
            $table->increments('id');
            $table->string('label')->nullable();
            $table->decimal('price_per_issue')->nullable();
            $table->unsignedInteger('member_id')->index('subscriptions_member_id_foreign');
            $table->integer('interval_multiplier')->default(1);
            $table->enum('interval', ['days', 'weeks', 'months'])->default('months');
            $table->enum('status', ['pending', 'active', 'paused', 'expired', 'payment-failed', 'cancelled'])->default('pending');
            $table->boolean('can_be_paused')->default(true);
            $table->integer('max_number_of_issues')->nullable()->default(99);
            $table->date('expiry_date')->nullable();
            $table->date('next_issue_date')->nullable();
            $table->timestamps();
        });

        Schema::create('subscriptions_bundletypes', function (Blueprint $table) {
            $table->unsignedInteger('subscription_id')->unique('subscriptions_bundletypes__sub_id');
            $table->unsignedInteger('bundletype_id')->index('subscriptions_bundletypes_bundletype_id_foreign');

            $table->unique(['subscription_id', 'bundletype_id']);
        });

        Schema::create('subscriptions_orders', function (Blueprint $table) {
            $table->increments('id');
            $table->unsignedInteger('subscription_id');
            $table->integer('order_id')->index('subscriptions_orders_order_id_foreign');
            $table->timestamp('created_at')->useCurrent();

            $table->unique(['subscription_id', 'order_id'], 'subscriptions_orders__unique_sub_id_order_id');
        });

        Schema::create('tree_cache', function (Blueprint $table) {
            $table->integer('id', true);
            $table->string('setup', 512)->index('setup');
            $table->longText('object');
            $table->timestamp('timestamp')->useCurrentOnUpdate()->useCurrent()->index('timestamp');
        });

        Schema::create('trustpilot_settings', function (Blueprint $table) {
            $table->increments('id');
            $table->string('key')->unique();
            $table->string('value');
        });

        Schema::create('unio__queue_messages_from_unio', function (Blueprint $table) {
            $table->increments('id');
            $table->enum('entity_type', ['customer', 'order', 'message']);
            $table->unsignedBigInteger('entity_id');
            $table->timestamp('created_at')->useCurrent();
            $table->timestamp('processed_at')->nullable();
            $table->text('response')->nullable();
            $table->text('payload_received')->nullable();
            $table->enum('status', ['pending', 'processing', 'success', 'failed'])->default('pending');
        });

        Schema::create('unio__queue_messages_to_unio', function (Blueprint $table) {
            $table->increments('id');
            $table->enum('entity_type', ['member', 'order', 'message', 'exemption']);
            $table->unsignedInteger('entity_id');
            $table->timestamp('created_at')->useCurrent();
            $table->timestamp('processed_at')->nullable();
            $table->text('response')->nullable();
            $table->enum('status', ['pending', 'processing', 'success', 'failed'])->default('pending');
        });

        Schema::create('vat', function (Blueprint $table) {
            $table->integer('id', true);
            $table->integer('order')->index('order');
            $table->double('name');
            $table->timestamp('timestamp')->useCurrent();
        });

        Schema::create('warehouses', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name');
            $table->string('code')->unique();
            $table->integer('priority')->nullable();
            $table->timestamps();
        });

        Schema::create('warehouses_stock', function (Blueprint $table) {
            $table->increments('id');
            $table->string('epos_code')->index();
            $table->string('warehouse')->index();
            $table->integer('quantity');
            $table->string('ref')->index();
            $table->timestamps();
        });

        Schema::create('warning_items', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('warning_id');
            $table->integer('item_id');
            $table->timestamps();

            $table->index(['warning_id', 'item_id']);
        });

        Schema::create('warnings', function (Blueprint $table) {
            $table->increments('id');
            $table->text('content');
            $table->tinyInteger('warning_type');
            $table->timestamps();
        });

        Schema::create('worldpay_responses', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('order_id')->index('worldpay_responses_order_id_foreign');
            $table->string('type', 64);
            $table->text('response');
            $table->timestamps();
        });

        Schema::table('bundle_items', function (Blueprint $table) {
            $table->foreign(['bundle_type_item_id'], 'bundle_items__bundle_type_item_id')->references(['id'])->on('bundle_types_items')->onUpdate('CASCADE')->onDelete('CASCADE');
            $table->foreign(['bundle_id'])->references(['id'])->on('bundles')->onUpdate('CASCADE');
        });

        Schema::table('bundle_types', function (Blueprint $table) {
            $table->foreign(['assessment_form_id'], 'bundle_types__assessment_form_id')->references(['id'])->on('assessment_forms')->onUpdate('CASCADE')->onDelete('SET NULL');
        });

        Schema::table('bundle_types_items', function (Blueprint $table) {
            $table->foreign(['shop_item_size_id'], 'bundle_types_items__shop_item_size_id')->references(['id'])->on('items_sizes')->onUpdate('CASCADE')->onDelete('CASCADE');
            $table->foreign(['bundle_type_id'])->references(['id'])->on('bundle_types')->onUpdate('CASCADE')->onDelete('CASCADE');
            $table->foreign(['shop_item_id'])->references(['id'])->on('items')->onUpdate('CASCADE')->onDelete('CASCADE');
        });

        Schema::table('bundles', function (Blueprint $table) {
            $table->foreign(['type_id'])->references(['id'])->on('bundle_types')->onUpdate('CASCADE');
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::table('bundles', function (Blueprint $table) {
            $table->dropForeign('bundles_type_id_foreign');
        });

        Schema::table('bundle_types_items', function (Blueprint $table) {
            $table->dropForeign('bundle_types_items__shop_item_size_id');
            $table->dropForeign('bundle_types_items_bundle_type_id_foreign');
            $table->dropForeign('bundle_types_items_shop_item_id_foreign');
        });

        Schema::table('bundle_types', function (Blueprint $table) {
            $table->dropForeign('bundle_types__assessment_form_id');
        });

        Schema::table('bundle_items', function (Blueprint $table) {
            $table->dropForeign('bundle_items__bundle_type_item_id');
            $table->dropForeign('bundle_items_bundle_id_foreign');
        });

        Schema::dropIfExists('worldpay_responses');

        Schema::dropIfExists('warnings');

        Schema::dropIfExists('warning_items');

        Schema::dropIfExists('warehouses_stock');

        Schema::dropIfExists('warehouses');

        Schema::dropIfExists('vat');

        Schema::dropIfExists('unio__queue_messages_to_unio');

        Schema::dropIfExists('unio__queue_messages_from_unio');

        Schema::dropIfExists('trustpilot_settings');

        Schema::dropIfExists('tree_cache');

        Schema::dropIfExists('subscriptions_orders');

        Schema::dropIfExists('subscriptions_bundletypes');

        Schema::dropIfExists('subscriptions');

        Schema::dropIfExists('subscription_preferences');

        Schema::dropIfExists('subscription_items');

        Schema::dropIfExists('static_blocks_sizes');

        Schema::dropIfExists('static_blocks_instances');

        Schema::dropIfExists('static_blocks');

        Schema::dropIfExists('standard_template_letters');

        Schema::dropIfExists('slides');

        Schema::dropIfExists('share_buttons');

        Schema::dropIfExists('settings');

        Schema::dropIfExists('seo_settings');

        Schema::dropIfExists('seo_schema_templates');

        Schema::dropIfExists('seo_redirects');

        Schema::dropIfExists('seo_pages');

        Schema::dropIfExists('seo_headings');

        Schema::dropIfExists('seo_defaults');

        Schema::dropIfExists('seo_canonicals');

        Schema::dropIfExists('sagepay_responses');

        Schema::dropIfExists('realex_responses');

        Schema::dropIfExists('property_types');

        Schema::dropIfExists('properties');

        Schema::dropIfExists('private_prescription_medications');

        Schema::dropIfExists('private_prescription_dosages');

        Schema::dropIfExists('placeholder_images');

        Schema::dropIfExists('pages');

        Schema::dropIfExists('page_options');

        Schema::dropIfExists('page_list_item_data');

        Schema::dropIfExists('page_list_item');

        Schema::dropIfExists('page_list');

        Schema::dropIfExists('page_cache');

        Schema::dropIfExists('order_worldpay');

        Schema::dropIfExists('order_statuses');

        Schema::dropIfExists('order_status_email_text');

        Schema::dropIfExists('order_status');

        Schema::dropIfExists('order_shipment_items');

        Schema::dropIfExists('order_shipment');

        Schema::dropIfExists('order_refunds');

        Schema::dropIfExists('order_refund_items');

        Schema::dropIfExists('order_realex');

        Schema::dropIfExists('order_protx');

        Schema::dropIfExists('order_paypal');

        Schema::dropIfExists('order_note');

        Schema::dropIfExists('order_items_meta');

        Schema::dropIfExists('order_items');

        Schema::dropIfExists('order_info');

        Schema::dropIfExists('order_evopay');

        Schema::dropIfExists('order_discounts');

        Schema::dropIfExists('order_coupon');

        Schema::dropIfExists('order_contact');

        Schema::dropIfExists('order_archive_items');

        Schema::dropIfExists('order_address');

        Schema::dropIfExists('order');

        Schema::dropIfExists('nhs_unilexicon_products');

        Schema::dropIfExists('nhs_reminders');

        Schema::dropIfExists('nhs_prescriptions');

        Schema::dropIfExists('nhs_prescription_items');

        Schema::dropIfExists('nhs_members_nhs_medical_categories');

        Schema::dropIfExists('nhs_members');

        Schema::dropIfExists('nhs_member_prescriptions');

        Schema::dropIfExists('nhs_member_doctor_surgeries');

        Schema::dropIfExists('nhs_medications');

        Schema::dropIfExists('nhs_medical_categories');

        Schema::dropIfExists('nhs_dosages');

        Schema::dropIfExists('multisite__sites');

        Schema::dropIfExists('multisite__custom_entities');

        Schema::dropIfExists('multisite__base_entities');

        Schema::dropIfExists('messenger_threads');

        Schema::dropIfExists('messenger_templates');

        Schema::dropIfExists('messenger_participants');

        Schema::dropIfExists('messenger_messages_read');

        Schema::dropIfExists('messenger_messages');

        Schema::dropIfExists('members_items_preferences');

        Schema::dropIfExists('members_info');

        Schema::dropIfExists('members_addresses');

        Schema::dropIfExists('members');

        Schema::dropIfExists('member_oauth_providers');

        Schema::dropIfExists('medications');

        Schema::dropIfExists('medical_conditions');

        Schema::dropIfExists('log_id_checks');

        Schema::dropIfExists('listing_cache');

        Schema::dropIfExists('languages');

        Schema::dropIfExists('keywords');

        Schema::dropIfExists('judopay_payments');

        Schema::dropIfExists('items_sizes_custom');

        Schema::dropIfExists('items_sizes');

        Schema::dropIfExists('items_seo_schema');

        Schema::dropIfExists('items_properties');

        Schema::dropIfExists('items_keywords');

        Schema::dropIfExists('items_ingredients');

        Schema::dropIfExists('items_info');

        Schema::dropIfExists('items_incompatible');

        Schema::dropIfExists('items_images');

        Schema::dropIfExists('items_faq');

        Schema::dropIfExists('items_custom_multiple');

        Schema::dropIfExists('items_custom');

        Schema::dropIfExists('items_complimented');

        Schema::dropIfExists('items_categories');

        Schema::dropIfExists('items_brands');

        Schema::dropIfExists('items__out_of_stock_notifications');

        Schema::dropIfExists('items__display_tags');

        Schema::dropIfExists('items');

        Schema::dropIfExists('item_sizes__restrictions');

        Schema::dropIfExists('item_size_labels');

        Schema::dropIfExists('item_delivery_rates');

        Schema::dropIfExists('inventories');

        Schema::dropIfExists('ingredients');

        Schema::dropIfExists('incompatible_medications');

        Schema::dropIfExists('incompatible_categories');

        Schema::dropIfExists('incompatibilities');

        Schema::dropIfExists('forms_responses_answers');

        Schema::dropIfExists('forms_responses');

        Schema::dropIfExists('forms_questions_options');

        Schema::dropIfExists('forms_questions_emails');

        Schema::dropIfExists('forms_questions');

        Schema::dropIfExists('forms_fieldsets');

        Schema::dropIfExists('forms');

        Schema::dropIfExists('feefo_reviews_settings');

        Schema::dropIfExists('feefo_reviews_logs');

        Schema::dropIfExists('featured_products');

        Schema::dropIfExists('evopay_responses');

        Schema::dropIfExists('emails_queued');

        Schema::dropIfExists('emails_lists_mails');

        Schema::dropIfExists('emails_lists');

        Schema::dropIfExists('emails_double_optin');

        Schema::dropIfExists('doctor_surgeries');

        Schema::dropIfExists('doctor_policies');

        Schema::dropIfExists('discountoffers');

        Schema::dropIfExists('delivery_zones');

        Schema::dropIfExists('delivery_services_options');

        Schema::dropIfExists('delivery_methods');

        Schema::dropIfExists('delivery_method_rates');

        Schema::dropIfExists('delivery_couriers_queue');

        Schema::dropIfExists('customers_also_viewed');

        Schema::dropIfExists('custom_fields_sets_fields');

        Schema::dropIfExists('custom_fields_sets');

        Schema::dropIfExists('custom_fields');

        Schema::dropIfExists('currencies');

        Schema::dropIfExists('coupons_conditions');

        Schema::dropIfExists('coupons_codes');

        Schema::dropIfExists('coupons');

        Schema::dropIfExists('country_states');

        Schema::dropIfExists('countries');

        Schema::dropIfExists('category_restricted_zones');

        Schema::dropIfExists('category_item_comparison');

        Schema::dropIfExists('categories__info_sections');

        Schema::dropIfExists('categories__display_tags');

        Schema::dropIfExists('categories');

        Schema::dropIfExists('catalog_import');

        Schema::dropIfExists('bundles');

        Schema::dropIfExists('bundle_types_items');

        Schema::dropIfExists('bundle_types');

        Schema::dropIfExists('bundle_items');

        Schema::dropIfExists('brands');

        Schema::dropIfExists('blog_categories');

        Schema::dropIfExists('black_list');

        Schema::dropIfExists('basket_recovery_sent_emails');

        Schema::dropIfExists('basket_recovery_logs');

        Schema::dropIfExists('basket_recovery');

        Schema::dropIfExists('basket_protx');

        Schema::dropIfExists('basket_items');

        Schema::dropIfExists('basket_info');

        Schema::dropIfExists('basket_address');

        Schema::dropIfExists('basket');

        Schema::dropIfExists('audit_log');

        Schema::dropIfExists('asset_gallery__icons');

        Schema::dropIfExists('assessments');

        Schema::dropIfExists('assessment_questions');

        Schema::dropIfExists('assessment_qualifying_answers__groups_config');

        Schema::dropIfExists('assessment_qualifying_answers__groups');

        Schema::dropIfExists('assessment_qualifying_answers__group_templates');

        Schema::dropIfExists('assessment_qualifying_answers');

        Schema::dropIfExists('assessment_forms');

        Schema::dropIfExists('assessment_answers_magento');

        Schema::dropIfExists('assessment_answers');

        Schema::dropIfExists('approval_log');

        Schema::dropIfExists('admin_users__linked_users');

        Schema::dropIfExists('admin_users__favourite_links');

        Schema::dropIfExists('admin_users');

        Schema::dropIfExists('admin_user_policy_confirmations');

        Schema::dropIfExists('admin_user_allowed_categories');

        Schema::dropIfExists('admin_roles');

        Schema::dropIfExists('admin_role_permissions');

        Schema::dropIfExists('admin_permissions');

        Schema::dropIfExists('admin_menu');
    }
};
