<?php

namespace Mtc\Plugins\MtcUnioIntegration\Classes;

use Mtc\Modules\Members\Models\Member;
use Mtc\Modules\MtcUnio\Classes\UnioApi;
use Mtc\Modules\MtcUnio\Interfaces\UnioApiRequestable;
use Mtc\Plugins\MembersMessaging\Classes\Message;
use Mtc\Shop\Order;

class QueueMessageToUnio extends QueueMessage implements UnioApiRequestable
{

    protected $table = 'unio__queue_messages_to_unio';

    // We don't have 'updated_at' field in the table.
    public $timestamps = false;

    protected $fillable = [
        'entity_type',
        'entity_id',
    ];

    const ENTITY_MEMBER = 'member';
    const ENTITY_ORDER = 'order';
    const ENTITY_MESSAGE = 'message';
    const ENTITY_EXEMPTION = 'exemption';


    public static function isEntityReadyToBeSynced($entity_type, $entity)
    {
        $is_ready = false;

        switch ($entity_type) {

            case self::ENTITY_MEMBER:
                if ($entity->is_activated || $entity->referrer) {
                    $is_ready = true;
                }
                break;

            case self::ENTITY_ORDER:
                $is_ready = true;
                break;

            case self::ENTITY_MESSAGE;
                $is_ready = true;
                break;

            case self::ENTITY_EXEMPTION;
                $entity->load('nhs_member');
                if ($entity->nhs_member->exemption_proof_file) {
                    $is_ready = true;
                }
                break;

        }

        return $is_ready;
    }


    public function saveResponse(string $response) : void
    {
        $this->processed_at = date('Y-m-d H:i');
        $this->response = $response;
        $this->save();
    }


    public function processResponse() : void
    {
        // TODO
    }


    public function getPayloadData() : array
    {
        $payload = null;

        switch ($this->entity_type) {

            case self::ENTITY_MEMBER:
                $member = Member::query()
                    ->with('addresses')
                    ->with('nhs_member.doctor_surgery')
                    ->with('nhs_member.medical_categories')
                    ->with('referrer.nhs_member.doctor_surgery')
                    ->where('id', $this->entity_id)
                    ->first()
                ;

                if ($member->is_email_system_generated()) {
                    $member->email = null;
                }

                $payload = $member->toArray();
                break;

            case self::ENTITY_ORDER:
                $order = Order::query()
                    ->where('id', $this->entity_id)
                    ->with('status')
                    ->with('items.prescription_item')
                    ->with('info')
                    ->with('customer.nhs_member')
                    ->with('delivery')
                    ->with('address')
                    ->with('notes')
                    ->with('assessments')
                    ->first()
                ;
                $payload = $order->toArray();
                break;

            case self::ENTITY_MESSAGE:
                $message = Message::query()
                    ->where('id', $this->entity_id)
                    ->with('thread.order')
                    ->with('thread.participants')
                    ->with('member')
                    ->with('admin_user')
                    ->first()
                ;
                $payload = $message->toArray();
                break;

            case self::ENTITY_EXEMPTION:
                $member = Member::query()
                    ->where('id', $this->entity_id)
                    ->first()
                ;
                $file_data = $member->nhs_member->get_exemption_proof_file_data();
                $payload = [
                    'customerId' => $member->id,
                    'data' => $file_data['content'],
                    'mimeType' => $file_data['mime_type'],
                ];
                break;

        }

        return $payload;
    }


    public function getQueryParams() : array
    {
        return [];
    }


    public function getEndpointName() : string
    {
        $endpoint_name = '';

        switch ($this->entity_type) {
            case self::ENTITY_MEMBER:
                $endpoint_name = UnioApi::ENDPOINT_CUSTOMER;
                break;

            case self::ENTITY_ORDER:
                $endpoint_name = UnioApi::ENDPOINT_ORDER;
                break;

            case self::ENTITY_MESSAGE:
                $endpoint_name = UnioApi::ENDPOINT_MESSAGE_SEND;
                break;

            case self::ENTITY_EXEMPTION:
                $endpoint_name = UnioApi::ENDPOINT_UPLOAD_EXEMPTION;
                break;
        }

        return $endpoint_name;
    }


    public function getMethodName() : string
    {
        return 'POST';
    }
}
