<?php

namespace Mtc\Plugins\SiteMenu\Controller;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Mtc\Plugins\SiteMenu\Classes\SiteMenu;

/**
 * @class SiteMenuAdmin
 * @extends Controller
 * @description Site Menu Admin Controller
 * */
class SiteMenuAdminController extends Controller
{
    public function index() : string
    {
        return json_encode(['success' => 'Site menu Admin Controller loaded successfully']);
    }

    /**
     * @description get site menu by id
     * @param $id int
     * @return \Illuminate\Http\JsonResponse
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @date  2025-05-22
     * @version 2.0
     * @copyright MTC Media
     */
    public function single(Request $request)
    {
        $id = $request->input('id');
        try {
            $site_menu = SiteMenu::find($id);
            if ($site_menu) {
                // set response code and return JSON response
                http_response_code(200);
                return json_encode($site_menu);
            } else {
                // set response code and return error message
                http_response_code(404);
                return json_encode(['error' => 'Site menu not found']);
            }
        }
        catch (\Exception $e) {
            // set response code and return error message
            http_response_code(500);
            return json_encode(['error' => 'An error occurred while fetching the site menu: ' . $e->getMessage()]);
        }
    }

    /**
     * @description get all site menus
     * @return String
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @date  2025-05-22
     * @version 2.0
     * @copyright MTC Media
     */
    public function all(Request $request) : string
    {
        try {
            $site_menu = SiteMenu::all();
            if ($site_menu) {
                // set response code and return JSON response
                http_response_code(200);
                return json_encode($site_menu);
            } else {
                // set response code and return error message
                http_response_code(404);
                return json_encode(['error' => 'Site menu not found']);
            }
        }
        catch (\Exception $e) {
            // set response code and return error message
            http_response_code(500);
            return json_encode(['error' => 'An error occurred while fetching the site menus: ' . $e->getMessage()]);
        }
    }

    /**
     * @description get site menu parent by id
     * @param $id int
     * @return String
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @date  2025-05-22
     * @version 2.0
     * @copyright MTC Media
     */

    public function parent(Request $request) : string {
        // get the id from the request
        $id = $request->input('id');
        try {
            $parent = SiteMenu::find($id)->parent_id;

            if ($parent != 0) {
                if ($parent) {
                    $parent_menu = SiteMenu::find($parent);
                    http_response_code(200);
                    return json_encode($parent_menu);
                } else {
                    http_response_code(404);
                    return json_encode(['error' => 'Parent site menu not found']);
                }
            }
            else {
                // return response code 200 with error message
                http_response_code(200);
                return json_encode(['error' => 'This is a root menu item, it has no parent']);
            }
        } catch (\Exception $e) {
            // set response code and return error message
            http_response_code(500);
            return json_encode(['error' => 'An error occurred while fetching the parent site menu: ' . $e->getMessage()]);
        }
    }

    /**
     * @method children
     * @description get site menu children by parent id
     * @param $id int
     * @return String
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @date  2025-05-22
     * @version 2.0
     * @copyright MTC Media
     */
    public function children(Request $request) : String
    {
        // get the id from the request
        $id = $request->input('id');
        try {
            $children = SiteMenu::where('parent_id', $id)->orderBy('order')->get();
            if ($children) {
                // set response code and return JSON response
                http_response_code(200);
                return json_encode($children);
            } else {
                // set response code and return error message
                http_response_code(404);
                return json_encode(['error' => 'Site menu not found']);
            }
        }
        catch (\Exception $e) {
            // set response code and return error message
            http_response_code(500);
            return json_encode(
                ['error' => 'An error occurred while fetching the site menu children: ' . $e->getMessage()]
            );
        }
    }

    /**
     * @method orderChange
     * @description swap the order of two site menu items
     * @param $id int
     * @return String
     * @author      Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @date        2025-07-22
     * @version     2.0
     * @copyright   MTC Media
     * */

    public function orderChange(Request $request) : String
    {
        $order = $request->input('order');


        try {
            if (isset($order) && is_array($order)) {
                foreach ($order as $item) {
                    $id = (int)$item['id'];
                    $position = (int)$item['position'];

                    // update the order of the site menu item
                    SiteMenu::where('id', $id)->update(['order' => $position]);
                }

                // set response code and return JSON response
                http_response_code(200);
                return json_encode(['status' => 'success']);
            } else {
                // set response code and return error message
                http_response_code(400);
                return json_encode(['status' => 'error', 'message' => 'Invalid data']);
            }

        } catch (\Exception $e) {
            http_response_code(500);
            return json_encode(['error' => 'An error occurred while swapping site menu items: ' . $e->getMessage()]);
        }
    }
}
