<?php

use Mtc\Shop\Category;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use MtcPharmacy\Multisite\Classes\MultisiteConstants;
use MtcPharmacy\Multisite\Classes\MultisiteManager;

/**
 * A place for category related helper functions to live.
 *
 * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
 * @author Lukas Giegerich <lukas.giegerich@mtcmedia.co.uk>
 */
class CategoryHelper {
    /**
     * Tracks selected category
     * @var int
     */
    public static $active_category;


    public static function getCategoryTree($sub_id = 0, $depth = PHP_INT_MAX)
    {
        $category_tree = [];

        if ($depth === 0) {
            return $category_tree;
        }

        $categories_query = Category::query()
            ->selectRaw('categories.*')
            ->active()
            ->orderBy('order', 'ASC')
            ->orderBy('name', 'ASC')
            ->where('sub_id', $sub_id)
        ;

        MultisiteManager::decorateQueryBaseOnly($categories_query, MultisiteConstants::ENTITY_TYPE_SHOP_CATEGORY);

        $categories = $categories_query->get();

        foreach ($categories as $category) {
            $category_tree[] = [
                'id' => $category->id,
                'name' => $category->name,
                'url' => $category->url,
                'sub_tree' => CategoryHelper::getCategoryTree($category->id, $depth),
            ];
        }

        return $category_tree;
    }

    /**
     * CategoryHelper::getCategoryParent()
     *
     * Finds a sub ID or returns false
     *
     * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
     * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
     * @param int $category_id
     * @return int $category_id bool
     *
     */
    private static function getCategoryParent($category_id)
    {
        try {
            return Category::findOrFail($category_id)->sub_id;
        } catch (ModelNotFoundException $e) {
            return false;
        }
    }

    /**
     * CategoryHelper::topCategory()
     *
     * Returns the root parent id of a category.
     *
     * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
     * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
     * @param  int $category_id child category ID
     * @return int $category_id root category ID
     */
    private static function topCategory($category_id)
    {
        $all_category_parents = Category::allParents($category_id);

        if (!empty($all_category_parents)) {
            return $all_category_parents[0];
        }

        return $category_id;
    }

    /**
     * CategoryHelper::generateBreadcrumb()
     *
     * Generate crumb of category
     *
     * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
     * @param  int $category_id
     * @return Array $crumb
     */
    public static function generateBreadcrumb($category_id)
    {

        $cat_data = category_data($category_id);

        $crumb = [
            'id' => 'cat_' . $cat_data['id'],
            'url' => browse_url($cat_data['id']),
            'name' => $cat_data['name'],
        ];

        return $crumb;

    }

    /**
     * CategoryHelper::generateBreadcrumbs()
     *
     * Generate Breadcrumb Trail For Categories.
     *
     * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
     * @param  int $category_id
     * @param  Array $breadcrumbs
     * @return Array $category_id
     */
    public static function generateBreadcrumbs($category_id)
    {

        $top_category = self::topCategory($category_id);

        $crumbs = [];

        while ($category_id != $top_category) {
            $crumbs[] = self::generateBreadcrumb($category_id);
            $category_id = self::getCategoryParent($category_id);
        }

        $crumbs[] = self::generateBreadcrumb($category_id);

        $crumbs = array_reverse($crumbs);

        return $crumbs;

    }

}
